/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.bpm.console.client.process;

import com.gwtext.client.data.Record;
import com.gwtext.client.widgets.PaddedPanel;
import com.gwtext.client.widgets.Panel;
import com.gwtext.client.widgets.TabPanel;
import com.gwtext.client.widgets.grid.RowSelectionModel;
import com.gwtext.client.widgets.grid.event.RowSelectionListenerAdapter;
import com.gwtext.client.widgets.layout.ColumnLayout;
import com.gwtext.client.widgets.layout.ColumnLayoutData;
import org.jboss.bpm.console.client.Editor;
import org.jboss.bpm.console.client.MainView;
import org.jboss.bpm.console.client.UIConstants;
import org.jboss.bpm.console.client.util.ModelModificationCallback;
import org.jboss.bpm.console.client.util.ModelListenerRegistry;
import org.jboss.bpm.console.client.util.ModelChangeListener;
import org.jboss.bpm.console.client.model.ProcessDefinitionRef;
import org.jboss.bpm.console.client.model.ProcessInstanceRef;
import org.jboss.bpm.console.client.widgets.*;

/**
 * @author Heiko.Braun <heiko.braun@jboss.com>
 *
 * @see org.jboss.bpm.console.client.process.ProcessDefinitionList#onExamine(int)
 */
public class ProcessInstanceListEditor extends Editor
{
   public final static String ID = "org.jboss.bpm.process.ProcessInstanceList";
   
   private MainView view;
   private ProcessDefinitionRef parent;
   private ProcessInstanceList instanceList;

   private Panel tokenEditorPanel;
   private ProcessInstanceRef selectedInstance;

   private ProcessInstanceDetailForm detailsForm;
   private TokenEditor currentTokenEditor;

   private TabPanel tabPanel;

   private ModelListenerRegistry modelListeners = new ModelListenerRegistry();
   private ModelModificationCallback modelModificationCallback;

   public ProcessInstanceListEditor(ProcessDefinitionRef proc, final MainView view)
   {
      super();

      this.parent = proc;
      this.view = view;
      this.setId( ProcessInstanceListEditor.createWidgetID(proc) );

      // --------------------------------------------------

      this.setLayout( new ColumnLayout() );
      this.setWidth(UIConstants.EDITOR_WIDTH);

      // ---------------

      Panel leftHand = new Panel();
      leftHand.setFrame(false);
      leftHand.setHeader(false);
      leftHand.setBorder(false);

      modelModificationCallback = new ModelModificationCallback()
      {
         public void onStaleModel()
         {
            resetEditor();
         }
      };

      instanceList = new ProcessInstanceList(modelModificationCallback, proc, "Process Instances", view);

      // ---------------

      tabPanel = new TabPanel();
      tabPanel.setPaddings(10);
      tabPanel.setPlain(true);
      tabPanel.setActiveTab(0);
      tabPanel.setWidth(UIConstants.EDITOR_PANEL_WIDTH);

      PaddedPanel tabPadding = new PaddedPanel(tabPanel, 0,10,0,10);

      // ----------------------------------------


      detailsForm = new ProcessInstanceDetailForm(view, modelModificationCallback);
      modelListeners.addListener( (ModelChangeListener) detailsForm);

      // ----------------


      final RowSelectionModel sm = new RowSelectionModel(true);
      sm.addListener(
            new RowSelectionListenerAdapter()
            {
               public void onRowSelect(RowSelectionModel sm, int rowIndex, Record record)
               {
                  selectedInstance = ProcessInstanceList.transform(record);

                  modelListeners.fireRecordChangeEvent(record);
                  modelListeners.fireModelChangeEvent(selectedInstance);

                  // --------------------

                  // token form update
                  // TODO: can't this use ModelChangeListener callbacks as well?
                  tokenEditorPanel.clear();
                  currentTokenEditor = new TokenEditor(view, selectedInstance, modelModificationCallback);
                  tokenEditorPanel.add(   // TODO: LRU caching (i.e. last 5)
                        currentTokenEditor
                  );
                  tokenEditorPanel.doLayout();
               }
            }
      );
      instanceList.setRowSelectionModel(sm);


      tokenEditorPanel = new Panel("Tokens");
      tokenEditorPanel.setIconCls("bpm-tools-icon");

      tabPanel.add( detailsForm );
      tabPanel.add(tokenEditorPanel);

      // ----------------------------------

      HelpPanel help = new HelpPanel(UIConstants.TEASER_PANEL_WIDTH, 200, "Managing process instances");

      TeaserPanel teaserPanel = new TeaserPanel();
      teaserPanel.add(help);

      // ----------------------------------

      leftHand.add(instanceList);
      leftHand.add(tabPadding);

      this.add(leftHand, new ColumnLayoutData(0.7) );
      this.add(teaserPanel , new ColumnLayoutData(0.3) );

   }

   public String getEditorId()
   {
      return getId();
   }

   public String getTitle()
   {
      return parent.getName()+" (#"+parent.getVersion()+")";
   }


   public String getIconCSS()
   {
      return "bpm-instance-icon";
   }

   void resetEditor()
   {
      instanceList.reloadStore();
      selectedInstance = null;

      modelListeners.fireResetEvent();
      
      if(currentTokenEditor!=null)
         currentTokenEditor.resetEditor();

      tokenEditorPanel.clear();
      tokenEditorPanel.doLayout();      
   }

   public static String createWidgetID(ProcessDefinitionRef proc)
   {
      return ProcessInstanceListEditor.ID+"."+proc.getName()+"."+proc.getVersion();

   }
}

