/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.bpm.console.client.process;

import org.jboss.bpm.console.client.util.ConsoleLog;
import com.google.gwt.http.client.*;
import com.gwtext.client.core.EventObject;
import com.gwtext.client.data.Node;
import com.gwtext.client.widgets.Panel;
import com.gwtext.client.widgets.layout.ColumnLayout;
import com.gwtext.client.widgets.layout.ColumnLayoutData;
import com.gwtext.client.widgets.tree.TreeNode;
import com.gwtext.client.widgets.tree.TreePanel;
import com.gwtext.client.widgets.tree.event.TreeNodeListenerAdapter;
import org.jboss.bpm.console.client.MainView;
import org.jboss.bpm.console.client.UIConstants;
import org.jboss.bpm.console.client.util.ModelModificationCallback;
import org.jboss.bpm.console.client.model.ProcessInstanceRef;
import org.jboss.bpm.console.client.model.jbpm3.TokenReference;

/**
 * @author Heiko.Braun <heiko.braun@jboss.com>
 */
public class TokenEditor extends Panel
{

   private TokenForm tokenForm;
   private TokenTree tokenTree;
   private MainView mainView;
   private ModelModificationCallback callback;

   public TokenEditor(
         MainView mainView, final ProcessInstanceRef instance,
         ModelModificationCallback callback
   )
   {
      super();

      this.mainView = mainView;
      this.callback = callback;

      this.setHeader(false);
      this.setBorder(false);
      this.setFrame(false);

      this.setLayout(new ColumnLayout());

      // ----------------

      tokenTree = new TokenTree(instance.getRootToken(),
            new TokenSelectionCallback()
            {
               public void onSelectedToken(TokenReference tok)
               {
                  // update form
                  tokenForm.display(instance, tok);
               }
            }
      );

      tokenTree.expandAll();

      // ----------------

      tokenForm = new TokenForm(
            new SignalCallback()
            {

               public void onSignalToken(TokenReference tok, String signal)
               {
                  remoteSignal(tok, signal);
               }
            }
      );

      // ----------------

      this.add(tokenTree, new ColumnLayoutData(0.3));
      this.add(tokenForm, new ColumnLayoutData(0.7));
   }

   public void resetEditor()
   {
      tokenForm.resetForm();
   }

   class TokenTree extends TreePanel
   {
      private TokenSelectionCallback callback;

      public TokenTree(final TokenReference rootToken,
                       final TokenSelectionCallback callback)
      {
         this.callback = callback;

         this.setBorder(false);

         TreeNode outermost = new TreeNode("Tokens");
         TreeNode rootNode = buildTreeNode(rootToken);
         outermost.appendChild(rootNode);

         buildChildNodes(rootNode, rootToken);

         setRootVisible(true);
         setRootNode(outermost);
         outermost.setExpanded(true);
      }

      private void buildChildNodes(TreeNode parent, TokenReference rootToken)
      {
         for(final TokenReference childToken : rootToken.getChildren())
         {
            TreeNode child = buildTreeNode(childToken);
            buildChildNodes(child, childToken); // recursive
            parent.appendChild(child);
         }
      }

      private TreeNode buildTreeNode(final TokenReference tok)
      {
         TreeNode treeNode = new TreeNode("Token " +tok.getId());
         treeNode.setExpanded(true);
         treeNode.addListener(
               new TreeNodeListenerAdapter()
               {
                  public void onClick(Node node, EventObject eventObject)
                  {
                     callback.onSelectedToken(tok);
                  }
               }
         );

         if(!tok.canBeSignaled())
            treeNode.setIcon("images/icons/lock.png");
         return treeNode;
      }
   }

   interface TokenSelectionCallback
   {
      void onSelectedToken(TokenReference tok);
   }

   public interface SignalCallback
   {
      void onSignalToken(TokenReference tok, String signal);
   }

   private void remoteSignal(TokenReference tok, String signal)
   {

      String url = signal.equals(UIConstants.DEFAULT_TRANSITION) ?
            mainView.getUrlBuilder().getTokenSignalUrl(tok) :
            mainView.getUrlBuilder().getTokenSignalUrl(tok, signal);

      RequestBuilder rb = new RequestBuilder(RequestBuilder.POST, url);

      try
      {
         rb.sendRequest(null,
               new RequestCallback() {

                  public void onResponseReceived(Request request, Response response)
                  {
                     if(response.getStatusCode()!=200)
                        ConsoleLog.error("Signaling failed: " +response.getText());
                     else
                     {
                        resetEditor();
                        callback.onStaleModel();
                     }
                  }

                  public void onError(Request request, Throwable t)
                  {
                     // Unknown error
                     ConsoleLog.error("Unknown error", t);
                  }
               });
      }
      catch (RequestException e1)
      {
         ConsoleLog.error("Request failed", e1);
      }
   }
}

