/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.bpm.console.client.task;

import org.jboss.bpm.console.client.util.ConsoleLog;
import org.jboss.bpm.console.client.util.ModelListenerRegistry;
import com.google.gwt.http.client.*;
import com.google.gwt.json.client.JSONParser;
import com.google.gwt.json.client.JSONValue;
import com.gwtext.client.core.EventObject;
import com.gwtext.client.data.Record;
import com.gwtext.client.widgets.*;
import com.gwtext.client.widgets.event.ButtonListenerAdapter;
import com.gwtext.client.widgets.event.PanelListenerAdapter;
import com.gwtext.client.widgets.form.Form;
import com.gwtext.client.widgets.form.FormPanel;
import org.jboss.bpm.console.client.MainView;
import org.jboss.bpm.console.client.model.DTOParser;
import org.jboss.bpm.console.client.model.TaskRef;
import org.jboss.bpm.console.client.widgets.FormWidgets;
import org.jboss.bpm.console.client.util.ModelChangeListener;
import org.jboss.bpm.console.client.widgets.RefreshableComboBox;

import java.util.List;

/**
 * Form panel for task assignments.
 *
 * @author Heiko.Braun <heiko.braun@jboss.com>
 */
class AssignmentFormPanel extends Panel implements ModelChangeListener
{
   private FormPanel formPanel;

   private MainView mainView;
   private AssignmentCallback callback;

   private List<String> availableActors;
   private TaskRef selectedInstance;

   public AssignmentFormPanel(MainView mainView, AssignmentCallback callback)
   {
      super();
      this.mainView = mainView;
      this.callback = callback;

      setTitle("Reassign Task");
      setBorder(false);
      setFrame(false);
      setIconCls("bpm-tools-icon");

      // ---------------

      formPanel = FormWidgets.createBaseFormPanel();
      RefreshableComboBox cb = new RefreshableComboBox("actorId", "Actor");
      formPanel.add(cb);

      final Button assignBtn = new Button("Delegate",
            new ButtonListenerAdapter()
            {

               public void onClick(Button button, EventObject eventObject)
               {
                  Form form = formPanel.getForm();
                  RefreshableComboBox cb = (RefreshableComboBox)form.findField("actorId");
                  String rawValue = cb.getRawValue();

                  if(null==rawValue || rawValue.equals(""))
                  {
                     MessageBox.alert("Please select an actor.");
                     return;
                  }

                  if(null==selectedInstance)
                     throw new IllegalArgumentException("Task selection cannot be null");

                  reassignTask(selectedInstance.getId(), rawValue);  
               }
            }
      );

      formPanel.add(assignBtn);
      this.add(formPanel);

      // ---------------
      this.addListener(
            new PanelListenerAdapter()
            {
               public void onAfterLayout(Container container)
               {
                  refreshComboBox();
               }
            }
      );

   }


   public void onModelChange(ModelListenerRegistry parent, Object changeEvent)
   {      
      selectedInstance= (TaskRef)changeEvent;
      availableActors = null;

      boolean hasPooledActors = selectedInstance.getPooledActors().size()>0;
      if(hasPooledActors)
      {
         // has pooled actors -> can be asigned to pooled actors only
         availableActors = selectedInstance.getPooledActors();
      }
      else if(!hasPooledActors &&
            TaskRef.STATE.ASSIGNED ==selectedInstance.getCurrentState())
      {
         // no pooled actors but an assigned actor
         loadActors();
      }
      else
      {
         // no pooled actors and no assignee
         // should be covered by assignment handler
         throw new IllegalStateException(selectedInstance + " has no pooled actors and no assignee");
      }

      // finally refresh the combobox
      refreshComboBox();
   }


   public void onRecordChange(ModelListenerRegistry parent, Record record)
   {
      // ignore
   }

   private void loadActors()
   {
      String url = mainView.getUrlBuilder().getAvailableActorsUrl(mainView.getUsername());
      RequestBuilder rb = new RequestBuilder(RequestBuilder.GET, url );

      try
      {
         rb.sendRequest(null,
               new RequestCallback() {

                  public void onResponseReceived(Request request, Response response)
                  {
                     if(200 != response.getStatusCode())
                        ConsoleLog.error("Failed to retrieve groups: " + response.getStatusText());

                     JSONValue value = JSONParser.parse(response.getText());

                     availableActors = DTOParser.parseStringArray(value);                                             
                  }

                  public void onError(Request request, Throwable t)
                  {
                     // Unknown error
                     ConsoleLog.error("Unknown error", t);
                  }
               });
      }
      catch (RequestException e1)
      {
         ConsoleLog.error("Request failed" ,e1);
      }
   }

   private void refreshComboBox()
   {
      if(null== availableActors)
         throw new IllegalArgumentException("available actors cannot be null");

      if(isRendered())
      {
         Form form = formPanel.getForm();
         RefreshableComboBox cb = (RefreshableComboBox)form.findField("actorId");
         cb.display(availableActors);
      }
   }

   private void reassignTask(long taskId, String actor)
   {
      RequestBuilder rb = new RequestBuilder(RequestBuilder.POST,
            mainView.getUrlBuilder().getTaskAssignmentURL(taskId, actor)
      );

      try
      {
         rb.sendRequest("",
               new RequestCallback() {

                  public void onResponseReceived(Request request, Response response)
                  {
                     if(200 != response.getStatusCode())
                        ConsoleLog.error("Failed to reassign task: " + response.getStatusText());

                     selectedInstance = null;
                     callback.onAssignmentDone();
                  }

                  public void onError(Request request, Throwable t)
                  {
                     // Unknown error
                     ConsoleLog.error("Unknown error", t);
                  }
               });
      }
      catch (RequestException e1)
      {
         ConsoleLog.error("Request failed" ,e1);
      }
   }

   public void onReset()
   {
      Form form = formPanel.getForm();      
      form.reset();
   }
}
