/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.bpm.console.client.task;

import org.jboss.bpm.console.client.util.ConsoleLog;
import com.google.gwt.http.client.*;
import com.gwtext.client.core.EventObject;
import com.gwtext.client.data.Record;
import com.gwtext.client.widgets.Button;
import com.gwtext.client.widgets.MessageBox;
import com.gwtext.client.widgets.Panel;
import com.gwtext.client.widgets.event.ButtonListenerAdapter;
import com.gwtext.client.widgets.form.ComboBox;
import com.gwtext.client.widgets.form.Form;
import com.gwtext.client.widgets.form.FormPanel;
import com.gwtext.client.widgets.form.TextField;
import org.jboss.bpm.console.client.MainView;
import org.jboss.bpm.console.client.UIConstants;
import org.jboss.bpm.console.client.model.TaskRef;
import org.jboss.bpm.console.client.widgets.FormWidgets;
import org.jboss.bpm.console.client.util.ModelChangeListener;
import org.jboss.bpm.console.client.util.ModelListenerRegistry;
import org.jboss.bpm.console.client.widgets.RefreshableComboBox;

/**
 * @author Heiko.Braun <heiko.braun@jboss.com>
 */
class TaskDetailsFormPanel extends Panel implements ModelChangeListener
{

   private FormPanel formPanel;
   private TaskRef selectedInstance;
   private MainView mainView;

   private Button startBtn;
   private Button stopBtn;
   private Button endBtn;

   private AssignmentCallback callback;
   
   public TaskDetailsFormPanel(final MainView mainView, AssignmentCallback callback)
   {
      super();
      this.mainView = mainView;
      this.callback = callback;

      setTitle("Task Details");
      setBorder(false);
      setFrame(false);

      // ---------------

      formPanel = FormWidgets.createBaseFormPanel();

      //the field names must match the data field values from the Store
      TextField idField = new TextField("Id", "id", 230);
      idField.setReadOnly(true);
      formPanel.add(idField);

      TextField nameField = new TextField("Name", "name", 230);
      nameField.setReadOnly(true);
      formPanel.add(nameField);

      TextField actorField = new TextField("Actor", "actor", 230);
      actorField.setReadOnly(true);
      formPanel.add(actorField);

      TextField stateField = new TextField("State", "currentState", 230);
      stateField.setReadOnly(true);
      formPanel.add(stateField);

      // ----------
      RefreshableComboBox cb = new RefreshableComboBox("signal", "Signal");
      formPanel.add(cb);

      startBtn = new Button("Claim",
            new ButtonListenerAdapter()
            {

               public void onClick(Button button, EventObject eventObject)
               {
                  if(null==selectedInstance)
                  {
                     MessageBox.alert("Please select a task.");
                     return;
                  }
                  selectedInstance.setActor(mainView.getUsername());
                  reassignTask();
               }
            }
      );

      stopBtn = new Button("Release",
            new ButtonListenerAdapter()
            {

               public void onClick(Button button, EventObject eventObject)
               {
                  if(null==selectedInstance)
                  {
                     MessageBox.alert("Please select a task.");
                     return;
                  }
                  selectedInstance.setActor(null);
                  reassignTask();
               }
            }
      );

      endBtn = new Button("Complete",
            new ButtonListenerAdapter()
            {

               public void onClick(Button button, EventObject eventObject)
               {
                  if(null==selectedInstance)
                  {
                     MessageBox.alert("Please select a task.");
                     return;
                  }

                  selectedInstance.close();

                  ComboBox cb = (ComboBox) formPanel.getForm().findField("signal");
                  String signalName = cb.getRawValue();
                  if(signalName.equals(""))
                     MessageBox.alert("Please select a signal to end this task");
                  else
                     endTask(signalName);
               }
            }
      );

      formPanel.addButton(startBtn);
      formPanel.addButton(stopBtn);
      formPanel.addButton(endBtn);

      this.add(formPanel);
   }

   private void reassignTask()
   {
      RequestBuilder rb = new RequestBuilder(RequestBuilder.POST,
            mainView.getUrlBuilder().getTaskAssignmentURL(
                  selectedInstance.getId(), selectedInstance.getActor()
            )
      );

      try
      {
         rb.sendRequest("",
               new RequestCallback() {

                  public void onResponseReceived(Request request, Response response)
                  {
                     if(200 != response.getStatusCode())
                        ConsoleLog.error("Failed to reassign task: " + response.getStatusText());
                     
                     selectedInstance = null;
                     callback.onAssignmentDone();                     
                  }

                  public void onError(Request request, Throwable t)
                  {
                     // Unknown error
                     ConsoleLog.error("Unknown error", t);
                  }
               });
      }
      catch (RequestException e1)
      {
         ConsoleLog.error("Request failed" ,e1);
      }
   }

   private void endTask(String signalName)
   {
      String taskCloseURL = signalName.equals(UIConstants.DEFAULT_TRANSITION) ?
            mainView.getUrlBuilder().getTaskEndURL(selectedInstance.getId()) :
            mainView.getUrlBuilder().getTaskEndURL(selectedInstance.getId(), signalName);

      RequestBuilder rb = new RequestBuilder(RequestBuilder.POST,taskCloseURL);

      try
      {
         rb.sendRequest("",
               new RequestCallback() {

                  public void onResponseReceived(Request request, Response response)
                  {
                     if(200 != response.getStatusCode())
                        ConsoleLog.error("Failed to end task: " + response.getStatusText());
                     
                     selectedInstance = null;
                     callback.onTaskEnd();
                  }

                  public void onError(Request request, Throwable t)
                  {
                     // Unknown error
                     ConsoleLog.error("Unknown error", t);
                  }
               });
      }
      catch (RequestException e1)
      {
         ConsoleLog.error("Failed to close task", e1);
      }
   }


   public void onRecordChange(ModelListenerRegistry parent, Record changeEvent)
   {      
      selectedInstance = TaskList.transform(changeEvent);

      Form form = formPanel.getForm();
      RefreshableComboBox cb = (RefreshableComboBox)form.findField("signal");

      if(selectedInstance.getPooledActors().isEmpty())
      {
         // no group assignment available
         startBtn.disable();
         stopBtn.disable();
      }
      else if(TaskRef.STATE.ASSIGNED == selectedInstance.getCurrentState())
      {
         startBtn.disable();
         stopBtn.enable();
         cb.enable();
         endBtn.enable();
      }
      else
      {
         startBtn.enable();
         stopBtn.disable();
         cb.disable();
         endBtn.disable();
      }

      cb.display( selectedInstance.getTransitionNames() );

      formPanel.getForm().loadRecord(changeEvent);
      formPanel.doLayout();
   }


   public void onModelChange(ModelListenerRegistry parent, Object changeEvent)
   {
      // ignore
   }

   public void onReset()
   {
      formPanel.getForm().reset();
   }
}
