/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.bpm.console.client.task;

import org.jboss.bpm.console.client.util.ConsoleLog;
import org.jboss.bpm.console.client.util.ModelModificationCallback;
import com.google.gwt.core.client.JavaScriptObject;
import com.google.gwt.json.client.JSONObject;
import com.gwtext.client.core.EventObject;
import com.gwtext.client.core.Function;
import com.gwtext.client.core.SortDir;
import com.gwtext.client.data.*;
import com.gwtext.client.data.event.StoreListenerAdapter;
import com.gwtext.client.widgets.*;
import com.gwtext.client.widgets.event.PanelListenerAdapter;
import com.gwtext.client.widgets.grid.*;
import com.gwtext.client.widgets.grid.event.GridCellListener;
import org.jboss.bpm.console.client.MainView;
import org.jboss.bpm.console.client.UIConstants;
import org.jboss.bpm.console.client.model.DTOParser;
import org.jboss.bpm.console.client.model.TaskRef;

import java.util.HashMap;
import java.util.Map;

/** 
 * @author Heiko.Braun <heiko.braun@jboss.com>
 */
public class TaskList extends Panel
{
   private Map<Integer, TaskRef> row2taskref = new HashMap<Integer, TaskRef>();
   private MainView mainView;
   private GridPanel grid;
   private GroupingStore store;
   private String title;
   private RowSelectionModel rowSelectionModel;
   public static final int PAGE_SIZE = 15;
   private int selectedRowIndex = -1;

   private ModelModificationCallback modelModificationCallback;

   public TaskList(ModelModificationCallback callback, String titleName, final MainView view)
   {
      this.mainView = view;
      this.title = titleName;
      this.modelModificationCallback = callback;

      this.setPaddings(10);
      this.setHeader(false);
      this.setBorder(false);
      this.setFrame(false);
      this.setIconCls("bpm-task-icon");

      String resourceUrl = view.getUrlBuilder().getTaskListByActorURL(view.getUsername());
      DataProxy dataProxy = new ScriptTagProxy(resourceUrl, 1000*10);

      final Reader reader = createReader();

      store = new GroupingStore();
      store.setReader(reader);
      store.setDataProxy(dataProxy);
      store.setSortInfo(new SortState("id", SortDir.ASC));
      store.setGroupField("currentState");
      store.addStoreListener( new ListViewStoreListener(this) );

      // PanelListener will lazy load store data.
      // Store.onLoad() callback will populate the grid
      // and add it to the container panel.
      this.addListener(
            new PanelListenerAdapter()
            {
               public boolean doBeforeRender(Component component)
               {
                  modelModificationCallback.onStaleModel();
                  return true;
               }
            }
      );
   }

   protected void onRecordsLoaded(Record[] records)
   {
      try
      {
         int i=0;
         for(Record r : records)
         {
            TaskRef ref = transform(r);
            row2taskref.put(i, ref);
            i++;
         }
      }
      catch (Throwable e)
      {
         ConsoleLog.error("Failed to parse task ref", e);
      }

      ConsoleLog.debug("Loaded " + row2taskref.size() + " tasks");
   }

   public static TaskRef transform(Record r)
   {
      JavaScriptObject js = r.getDataAsJsObject();
      JSONObject jso = new JSONObject(js);
      TaskRef ref = DTOParser.parseTaskReference(jso);
      return ref;
   }

   private ColumnModel createColumnModel()
   {
      final ColumnModel columnModel = new ColumnModel(
            new ColumnConfig[]
                  {
                        new ColumnConfig("ID", "id", 35, true),
                        new ColumnConfig("Task Name", "name", 290, true, null, "expand"),
                        new ColumnConfig("Assigned to", "actor", 75, true),
                        new ColumnConfig("State", "currentState", 50, true)
                  }
      );
      return columnModel;
   }

   private JsonReader createReader()
   {
      final RecordDef recordDef = new RecordDef(
            new FieldDef[]{
                  new IntegerFieldDef("id"),
                  new StringFieldDef("name"),
                  new StringFieldDef("actor"),
                  new StringFieldDef("currentState")
            }
      );

      final JsonReader reader = new JsonReader(recordDef);
      reader.setRoot("tasks");
      reader.setTotalProperty("totalCount");
      reader.setId("id");
      return reader;
   }

   private class ListViewStoreListener extends StoreListenerAdapter
   {

      Panel containerPanel;

      public ListViewStoreListener(Panel containerPanel)
      {
         this.containerPanel = containerPanel;
      }

      /**
       * load callback will push records to implementation
       * and (!) add the grid to the container panel.
       */
      public void onLoad(Store store, Record[] records)
      {

         onRecordsLoaded(records);

         grid = assembleGrid();

         containerPanel.clear();
         containerPanel.add(grid);
         containerPanel.doLayout();
      }

      public void onLoadException(Throwable throwable)
      {
         ConsoleLog.error("Failed to load remote data", throwable);
      }

   }

   private GridPanel assembleGrid()
   {
      final ColumnModel columnModel = createColumnModel();

      GridPanel grid = new GridPanel();
      grid.setStore(store);
      grid.setColumnModel(columnModel);
      grid.setFrame(true);
      grid.setStripeRows(true);
      grid.setAutoExpandColumn("name");
      grid.setTitle(title);

      GroupingView gridView = new GroupingView();
      gridView.setForceFit(true);
      gridView.setGroupTextTpl("{text} ({[values.rs.length]} {[values.rs.length > 1 ?  \"Items\" : \"Item\"]})");

      if(rowSelectionModel!=null)
      {
         grid.setSelectionModel(rowSelectionModel);
         grid.doOnRender(new Function() {
            public void execute()
            {
               if(-1==selectedRowIndex)
                  rowSelectionModel.selectFirstRow();
               else
                  rowSelectionModel.selectRow(selectedRowIndex);
            }
         }, 10);
      }
      grid.setView(gridView);
      grid.setFrame(true);
      grid.setWidth(UIConstants.EDITOR_PANEL_WIDTH);
      grid.setHeight(250);
      grid.setEnableHdMenu(false);

      grid.setBottomToolbar(createToolbar(store));

      return grid;
   }

   private PagingToolbar createToolbar(final Store store)
   {
      PagingToolbar pagingToolbar = new PagingToolbar(store);
      pagingToolbar.setPageSize(PAGE_SIZE);
      pagingToolbar.setDisplayInfo(true);
      pagingToolbar.setDisplayMsg("{0} - {1} of {2}");

      pagingToolbar.addSeparator();
      pagingToolbar.addSpacer();
      
      return pagingToolbar;
   }


   public void reloadStore()
   {
      ConsoleLog.debug("Reload " + this.getId());
      store.load(0, PAGE_SIZE);
   }

   public GridPanel getGrid()
   {
      return grid;
   }

   public void setRowSelectionModel(RowSelectionModel model)
   {
      this.rowSelectionModel = model;
   }

   public class ListViewCellListener implements GridCellListener
   {
      public void onCellClick(GridPanel grid, int rowIndex, int colindex, EventObject e)
      {
         selectedRowIndex = rowIndex;
      }


      public void onCellContextMenu(GridPanel gridPanel, int i, int i1, EventObject eventObject)
      {

      }

      public void onCellDblClick(GridPanel gridPanel, int i, int i1, EventObject eventObject)
      {

      }
   }
}
