/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.bpm.console.client.task;

import com.gwtext.client.data.Record;
import com.gwtext.client.widgets.Component;
import com.gwtext.client.widgets.PaddedPanel;
import com.gwtext.client.widgets.Panel;
import com.gwtext.client.widgets.TabPanel;
import com.gwtext.client.widgets.event.PanelListenerAdapter;
import com.gwtext.client.widgets.grid.RowSelectionModel;
import com.gwtext.client.widgets.grid.event.RowSelectionListenerAdapter;
import com.gwtext.client.widgets.layout.ColumnLayout;
import com.gwtext.client.widgets.layout.ColumnLayoutData;
import org.jboss.bpm.console.client.Editor;
import org.jboss.bpm.console.client.MainView;
import org.jboss.bpm.console.client.UIConstants;
import org.jboss.bpm.console.client.model.TaskRef;
import org.jboss.bpm.console.client.util.ModelChangeListener;
import org.jboss.bpm.console.client.util.ModelListenerRegistry;
import org.jboss.bpm.console.client.widgets.TeaserPanel;
import org.jboss.bpm.console.client.util.ModelModificationCallback;

/**
 * @author Heiko.Braun <heiko.braun@jboss.com>
 */
public class TaskListEditor extends Editor implements AssignmentCallback
{

   public final static String ID = "org.jboss.bpm.task.TaskList";
   private MainView mainView;
   private TaskList taskList;
   private Panel teaserPanel;
   private boolean initialized;
   private TabPanel tabPanel;

   private ModelListenerRegistry modelListeners =new ModelListenerRegistry();

   public TaskListEditor(MainView mainView)
   {
      super();
      this.setId(ID);
      this.mainView = mainView;

      this.setLayout( new ColumnLayout() );
      this.setWidth(UIConstants.EDITOR_WIDTH);

      this.addListener(
            new PanelListenerAdapter()
            {
               public boolean doBeforeRender(Component component)
               {
                  return beforeRenderCallback();
               }
            }
      );

   }

   public String getEditorId()
   {
      return ID;
   }

   public String getTitle()
   {
      return "Tasks";
   }

   public String getIconCSS()
   {
      return "bpm-task-icon";
   }

   /**
    * TaskList editor requires an authenticated user,
    * hence the initialization through callback
    */
   public boolean beforeRenderCallback()
   {
      if(!initialized)
      {
         final Panel leftHand = new Panel();
         leftHand.setFrame(false);
         leftHand.setHeader(false);
         leftHand.setBorder(false);

         // ----------------------------------

         tabPanel = new TabPanel();
         tabPanel.setPaddings(10);
         tabPanel.setPlain(true);
         tabPanel.setActiveTab(0);
         tabPanel.setWidth(UIConstants.EDITOR_PANEL_WIDTH);

         PaddedPanel tabPadding = new PaddedPanel(tabPanel, 0,10,0,10);

         // ----------------------------------

         ModelModificationCallback modelModificationCallback = new ModelModificationCallback()
         {

            public void onStaleModel()
            {
               resetEditor();
            }
         };
         
         taskList = new TaskList(modelModificationCallback, "Task overview", mainView);
         final RowSelectionModel sm = new RowSelectionModel(true);
         sm.addListener(
               new RowSelectionListenerAdapter()
               {
                  public void onRowSelect(RowSelectionModel sm, int rowIndex, Record record)
                  {
                     TaskRef taskRef = TaskList.transform(record);
                     modelListeners.fireRecordChangeEvent(record);
                     modelListeners.fireModelChangeEvent(taskRef);
                  }
               }
         );
         taskList.setRowSelectionModel(sm);
         leftHand.add(taskList);

         // ----------------------------------

         Panel detailsPanel = new TaskDetailsFormPanel(mainView, this);
         Panel assignmentPanel = new AssignmentFormPanel(mainView, this);

         modelListeners.addListener((ModelChangeListener)detailsPanel);
         modelListeners.addListener((ModelChangeListener)assignmentPanel);

         tabPanel.add( detailsPanel );
         tabPanel.add( assignmentPanel );

         // ----------------------------------

         leftHand.add(tabPadding);

         // ----------------------------------

         teaserPanel = new TeaserPanel();

         // ----------------------------------

         this.add(leftHand, new ColumnLayoutData(0.7) );
         this.add(teaserPanel , new ColumnLayoutData(0.3) );

         doLayout();

         initialized = true;
      }

      return true;
   }

   private void resetEditor()
   {
      taskList.reloadStore();
      modelListeners.fireResetEvent();
   }

   public void onAssignmentDone()
   {
      resetEditor();
   }

   public void onTaskEnd()
   {
      resetEditor();
   }
}
