/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jbpm.calendar;

import java.io.Serializable;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.Properties;

/**
 * is a day on a business calendar.
 */
public class Day implements Serializable {

  private static final long serialVersionUID = 1L;

  DayPart[] dayParts;
  BusinessCalendar businessCalendar;

  public static Day[] parseWeekDays(Properties calendarProperties,
    BusinessCalendar businessCalendar) {
    DateFormat dateFormat = new SimpleDateFormat(calendarProperties.getProperty("hour.format"));
    String[] weekDayKeys = {
      null, "weekday.sunday", "weekday.monday", "weekday.tuesday", "weekday.wednesday",
      "weekday.thursday", "weekday.friday", "weekday.saturday"
    };

    Day[] weekDays = new Day[weekDayKeys.length];
    for (int i = Calendar.SUNDAY; i <= Calendar.SATURDAY; i++) {
      weekDays[i] = new Day(calendarProperties.getProperty(weekDayKeys[i]), dateFormat,
        businessCalendar);
    }
    return weekDays;
  }

  public Day(String dayPartsText, DateFormat dateFormat, BusinessCalendar businessCalendar) {
    this.businessCalendar = businessCalendar;

    String[] dayPartTexts = dayPartsText.split("[\\s&]+");
    if (dayPartTexts.length > 1 || dayPartTexts[0].length() > 0) {
      dayParts = new DayPart[dayPartTexts.length];
      for (int i = 0; i < dayParts.length; i++) {
        dayParts[i] = new DayPart(dayPartTexts[i], dateFormat, this, i);
      }
    }
    else {
      dayParts = new DayPart[0];
    }
  }

  public void findNextDayPartStart(int dayPartIndex, Date date, Object[] result) {
    // if there is a day part in this day that starts after the given date
    if (dayPartIndex < dayParts.length) {
      if (dayParts[dayPartIndex].isStartAfter(date)) {
        result[0] = dayParts[dayPartIndex].getStartTime(date);
        result[1] = dayParts[dayPartIndex];
      }
      else {
        findNextDayPartStart(dayPartIndex + 1, date, result);
      }
    }
    else {
      // descend recursively
      date = businessCalendar.findStartOfNextDay(date);
      Day nextDay = businessCalendar.findDay(date);
      nextDay.findNextDayPartStart(0, date, result);
    }
  }
}
