/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jbpm.web;

import java.io.IOException;
import java.io.PrintWriter;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.List;
import java.util.zip.ZipInputStream;
import org.apache.commons.fileupload.FileItem;
import org.apache.commons.fileupload.FileItemFactory;
import org.apache.commons.fileupload.FileUploadException;
import org.apache.commons.fileupload.disk.DiskFileItemFactory;
import org.apache.commons.fileupload.servlet.ServletFileUpload;

import org.jbpm.JbpmConfiguration;
import org.jbpm.JbpmContext;
import org.jbpm.graph.def.ProcessDefinition;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

public class ProcessUploadServlet extends javax.servlet.http.HttpServlet {

	private static final long serialVersionUID = 1L;

	private JbpmConfiguration jbpmConfiguration;

	public void init() throws ServletException {
		String jbpmCfgResource = getServletContext().getInitParameter("jbpm.configuration.resource");
		jbpmConfiguration = JbpmConfiguration.getInstance(jbpmCfgResource);
	}

	protected void doGet(HttpServletRequest request, HttpServletResponse response)
			throws ServletException, IOException {
		if (log.isTraceEnabled()) log.trace("handling status request");

		PrintWriter out = response.getWriter();
		writeHeader(out);
		out.println("<h3>Deploy a process</h3>");
		out.println("<form name='deploy-form' method='post' enctype='multipart/form-data'>");
		out.println("  <p>Process Archive: <input name='process-archive' type='file'/>");
		out.println("  <p><input name='deploy-button' type='submit' value='Deploy'/>");
		out.println("</form>");
		writeTrailer(out);
	}

	protected void doPost(HttpServletRequest request, HttpServletResponse response)
			throws ServletException, IOException {
		if (log.isTraceEnabled()) log.trace("handling upload request");

		FileItem processItem = parseRequest(request, response);
		if (processItem == null) return; // error has been sent

		JbpmContext jbpmContext = jbpmConfiguration.createJbpmContext();
		try {
			ProcessDefinition processDefinition = parseProcessArchive(processItem);
			deployProcessDefinition(processDefinition, response);
		}
		finally {
			jbpmContext.close();
		}
	}

	private void writeHeader(PrintWriter out) throws MalformedURLException {
		out.println("<html>");
		out.println("<head>");
		out.println("<title>Process Deployment</title>");
		URL css = getServletContext().getResource("/ua/jbpm.css");
		if (css != null) {
			out.print("<link rel='stylesheet' type='text/css' href='");
			out.print(css.toString());
			out.println("'/>");
		}
		out.println("</head>");
		out.println("<body>");
	}

	private void writeTrailer(PrintWriter out) {
		out.println("</body>");
		out.println("</html>");
	}

	private FileItem parseRequest(HttpServletRequest request, HttpServletResponse response)
			throws IOException {
		// check if request is multipart content
		if (!ServletFileUpload.isMultipartContent(request)) {
			response.sendError(HttpServletResponse.SC_BAD_REQUEST, "Not a multipart request");
			return null;
		}

		// Create a factory for disk-based file items
		FileItemFactory factory = new DiskFileItemFactory();
		// Create a new file upload handler
		ServletFileUpload upload = new ServletFileUpload(factory);
		try {
			// Parse the request
			List items = upload.parseRequest(request);
			if (items.isEmpty()) {
				response.sendError(HttpServletResponse.SC_BAD_REQUEST, "Request is empty");
				return null;
			}
			FileItem item = (FileItem) items.get(0);
			if (item.isFormField()) {
				response.sendError(HttpServletResponse.SC_BAD_REQUEST, "Field '" + item.getFieldName()
						+ "' is not an input file");
				return null;
			}
			return item;
		}
		catch (FileUploadException e) {
			log.error("failed to parse request", e);
			response.sendError(HttpServletResponse.SC_BAD_REQUEST, "Failed to parse request");
			return null;
		}
	}

	private ProcessDefinition parseProcessArchive(FileItem fileItem) throws IOException {
		if (log.isTraceEnabled()) log.trace("parsing process archive " + fileItem.getName());

		ZipInputStream processStream = new ZipInputStream(fileItem.getInputStream());
		try {
			ProcessDefinition processDefinition = ProcessDefinition.parseParZipInputStream(processStream);
			if (log.isTraceEnabled())
				log.trace("parsed process definition " + processDefinition.getName());
			return processDefinition;
		}
		finally {
			processStream.close();
		}
	}

	private void deployProcessDefinition(ProcessDefinition processDefinition,
			HttpServletResponse response) throws IOException {
		String processName = processDefinition.getName();
		JbpmContext jbpmContext = jbpmConfiguration.getCurrentJbpmContext();
		try {
			jbpmContext.deployProcessDefinition(processDefinition);

			if (log.isTraceEnabled()) log.trace("deployed process definition " + processName);
			PrintWriter out = response.getWriter();
			writeHeader(out);
			out.println("<h3>Deployment report</h3>");
			out.print("<p>Deployed process ");
			out.print(processName);
			out.println(" successfully</p>");
			writeTrailer(out);
		}
		catch (RuntimeException e) {
			jbpmContext.setRollbackOnly();

			log.error("failed to deploy process definition " + processName, e);
			response.sendError(HttpServletResponse.SC_INTERNAL_SERVER_ERROR,
					"Failed to deploy process");
		}
	}

	private static final Log log = LogFactory.getLog(ProcessUploadServlet.class);
}