/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.balalaika;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import javax.servlet.http.HttpServletRequest;
import javax.ws.rs.*;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.Response;
import java.io.*;
import java.util.HashMap;
import java.util.Enumeration;
import java.util.Map;

/**
 * BIRT integration facade.<p>
 *
 * Uses the server data directory (i.e. <code>$JBOSS_HOME/server/default/data/birt</code>)
 * as it's work directory (referred to as $WORK_DIR in the subsequent sections):
 *
 * <ul>
 *    <li>report template location: $WORK_DIR/
 *    <li>output directory: $WORK_DIR/output
 * </ul>
 *
 * NOTE: It requires a BIRT report engine to be installed in the work directory:
 * (<code>$WORK_DIR/ReportEngine</code>.
 *
 * @see org.jboss.balalaika.JMXServerConfig
 *
 * @author Heiko.Braun <heiko.braun@jboss.com>
 */
@Path("report")
public class ReportFacade
{
   private static final Log log = LogFactory.getLog(ReportFacade.class);
   private BirtService birtService;
   private boolean isInitialized;

   public void lazyInitialize(HttpServletRequest request)
   {
      if(!isInitialized)
      {
         JMXServerConfig jmxConfig = new JMXServerConfig();
         IntegrationConfig iConfig = new IntegrationConfig();

         File severDataDir = jmxConfig.getServerDataDir();
         String absServerDataDir = severDataDir.getAbsolutePath();

         String birtDataDir = absServerDataDir + "/birt";
         String defaultBirtHome = birtDataDir + "/ReportEngine";
         String birtOutputDir = birtDataDir + "/output";

         File birtOutput = new File(birtOutputDir);
         birtOutput.mkdirs(); // will create parent directoy as well

         // check dependency on ReportEngine
         if(! new File(defaultBirtHome).exists())
            throw new IllegalStateException("The BIRT report engine doesn't seem to be installed:" +defaultBirtHome);

         // --

         iConfig.setBirtHome(defaultBirtHome);
         iConfig.setOutputDir( birtOutputDir );
         iConfig.setReportDir( birtDataDir );

         StringBuffer sb = new StringBuffer();
         sb.append("http://");
         sb.append(request.getServerName()).append(":");
         sb.append(request.getServerPort());
         sb.append(request.getContextPath());
         sb.append(request.getServletPath());
         sb.append("/report/view/image");

         iConfig.setImageBaseUrl(sb.toString());

         log.info("BIRT home: " +iConfig.getBirtHome());
         log.info("Output dir: " +iConfig.getOutputDir());
         log.info("Report dir: " +iConfig.getReportDir());

         this.birtService = new BirtService(iConfig);
         this.birtService.create();

         isInitialized = true;
      }
   }

   @GET
   @Path("view/{fileName}")
   @Produces("text/html")
   public Response viewReportHtml(
         @PathParam("fileName")
         String fileName,
         @Context HttpServletRequest
               request
   )
   {
      lazyInitialize(request);

      Map queryParams = convertRequestParametersToMap(request);

      RenderMetaData renderMeta = new RenderMetaData();
      renderMeta.setReportName(fileName);
      renderMeta.setFormat(RenderMetaData.Format.HTML);
      renderMeta.setClassloader(Thread.currentThread().getContextClassLoader());
      renderMeta.getParameters().putAll(queryParams);
      
      String outputFileName = birtService.render(renderMeta);
      String absoluteFile = birtService.getIConfig().getOutputDir() + outputFileName;
      log.info("Render " + absoluteFile);

      File reportFile = new File(absoluteFile);
      return Response.ok(reportFile).type("text/html").build();
   }

   @GET
   @Path("view/image/{fileName}")
   public Response getImage(
         @PathParam("fileName")
         String fileName,
         @Context HttpServletRequest
               request
   )
   {
      lazyInitialize(request);
      
      String imageDir = birtService.getIConfig().getImageDirectory();
      String absName = imageDir + fileName;
      File imageFile = new File(absName);
      if(!imageFile.exists())
         throw new IllegalArgumentException("Image " +absName+" doesn't exist");
      return Response.ok(imageFile).build();
   }

   static public Map<String, String> convertRequestParametersToMap(HttpServletRequest request){
      Enumeration<?> initParameters=request.getParameterNames();
      HashMap<String, String> parameterMap = new HashMap<String, String>();
      while (initParameters.hasMoreElements()) {
         String param = (String) initParameters.nextElement();
         String value = request.getParameter(param);
         if (param != null & value != null)
            parameterMap.put(param, value);
      }
      return parameterMap;
   }
}
