/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software{} you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation{} either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY{} without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software{} if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.bpm.client;

//$Id: MessageManager.java 1984 2008-08-22 12:31:53Z thomas.diesler@jboss.com $

import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import javax.management.ObjectName;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jboss.bpm.model.Event;
import org.jboss.bpm.model.Message;
import org.jboss.bpm.model.Participant;
import org.jboss.bpm.model.Task;

/**
 * The ProcessEngine sends mesages through the MessageManager. <p/> A {@link Message} has an ID and is targeted to a
 * specific Participant. A component can register a {@link MessageListener} with the MessageManager.
 * 
 * @author thomas.diesler@jboss.com
 * @since 18-Jun-2008
 */
public abstract class MessageManager
{
  // provide logging
  private static final Log log = LogFactory.getLog(MessageManager.class);

  // The set of registered message listeners
  private Map<ObjectName, MessageListener> listeners = new HashMap<ObjectName, MessageListener>();

  /**
   * Locate the MessageManager
   */
  public static MessageManager locateMessageManager()
  {
    ProcessEngine engine = ProcessEngine.locateProcessEngine();
    return engine.getMessageManager();
  }

  /**
   * Add a MessageListener
   */
  public void addMessageListener(MessageListener listener)
  {
    synchronized (listeners)
    {
      if (hasMessageListener(listener.getID()))
        throw new IllegalStateException("Listener already registered: " + listener.getID());

      log.debug("addMessageListener: " + listener.getID());
      listeners.put(listener.getID(), listener);
    }
  }

  /**
   * Get the set of registered MessageListeners 
   */
  public Set<MessageListener> getMessageListeners()
  {
    synchronized (listeners)
    {
      HashSet<MessageListener> set = new HashSet<MessageListener>(listeners.values());
      return Collections.unmodifiableSet(set);
    }
  }

  /**
   * Get a MessageListener for a given ID
   * 
   * @return null if there is none
   */
  public MessageListener getMessageListener(ObjectName listenerID)
  {
    synchronized (listeners)
    {
      return listeners.get(listenerID);
    }
  }

  /**
   * True if there is a MessageListener for a given ID
   */
  public boolean hasMessageListener(ObjectName listenerID)
  {
    return getMessageListener(listenerID) != null;
  }

  /**
   * Remove an MessageListener
   */
  public void removeMessageListener(ObjectName listenerID)
  {
    synchronized (listeners)
    {
      log.debug("removeMessageListener: " + listenerID);
      listeners.remove(listenerID);
    }
  }

  /**
   * Send a message to a given {@link Task} or {@link Event}
   */
  public abstract void sendMessage(ObjectName procID, String targetName, Message msg);
  
  /**
   * Send a message to a given {@link MessageListener}
   */
  public void sendMessage(Message msg)
  {
    Participant toRef = msg.getToRef().getEntityRef();
    if (toRef == null)
      throw new IllegalArgumentException("Target entity cannot be null");
    
    MessageListener msgListener = getMessageListener(toRef.getName());
    if (msgListener == null)
      throw new IllegalStateException("No message listener registered for: " + toRef);

    log.debug("sendMessage to '" + toRef + "' => " + msg);
    msgListener.catchMessage(msg);
  }
}