/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.bpm.client;

// $Id: ProcessManager.java 1946 2008-08-20 22:09:12Z thomas.diesler@jboss.com $

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import javax.management.ObjectName;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jboss.bpm.EngineShutdownException;
import org.jboss.bpm.model.Process;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

/**
 * The process manager is the entry point to create, find and otherwise manage processes.
 * 
 * @author thomas.diesler@jboss.com
 * @since 18-Jun-2008
 */
public abstract class ProcessManager
{
  // provide logging
  private static final Log log = LogFactory.getLog(ProcessManager.class);

  // The map of registerd dialect handlers
  protected Map<String, DialectHandler> dialectHandlers;
  // The dialect registry
  protected DialectRegistry dialectRegistry;
  // The set of registered processes
  private Map<ObjectName, Process> procs = new HashMap<ObjectName, Process>();

  // Hide public constructor
  protected ProcessManager()
  {
  }

  /**
   * Locate the ProcessManager
   */
  public static ProcessManager locateProcessManager()
  {
    ProcessEngine engine = ProcessEngine.locateProcessEngine();
    return engine.getProcessManager();
  }

  /**
   * Create a Process from a XML string in one of the supported formats
   */
  public final Process createProcess(String pXML)
  {
    String nsURI = getNamespaceURI(new ByteArrayInputStream(pXML.getBytes()));
    Process proc = getDialectHandler(nsURI).createProcess(pXML, false);
    return proc;
  }

  /**
   * Create a Process from an URL to a XML descritor in one of the supported formats
   */
  public final Process createProcess(URL pURL) throws IOException
  {
    String nsURI = getNamespaceURI(pURL.openStream());
    Process proc = getDialectHandler(nsURI).createProcess(pURL, false);
    return proc;
  }

  /**
   * Get the set of registered Processes
   */
  public Set<Process> getProcesses()
  {
    Set<Process> procSet = new HashSet<Process>(procs.values());
    return Collections.unmodifiableSet(procSet);
  }

  /**
   * Find the set of Processes for a given name
   * 
   * @param name The process name
   * @param status The optional process status
   * @return An empty set if the process cannot be found
   */
  public Set<Process> getProcesses(String name, Process.ProcessStatus status)
  {
    Set<Process> procSet = new HashSet<Process>();
    for (Process aux : procs.values())
    {
      if (aux.getName().equals(name))
      {
        if (status == null || aux.getProcessStatus() == status)
          procSet.add(aux);
      }
    }
    return procSet;
  }

  /**
   * Get a Process for a given id
   */
  public Process getProcessByID(ObjectName procID)
  {
    Process proc = procs.get(procID);
    return proc;
  }

  /**
   * Register a Process.
   */
  public void registerProcess(Process proc)
  {
    if (ProcessEngine.locateProcessEngine().isPrepareForShutdown())
      throw new EngineShutdownException("Cannot register a process while engine is shutting down");
    
    log.debug("registerProcess: " + proc);
    procs.put(proc.getID(), proc);
  }

  /**
   * Unregister a Process.
   */
  public void unregisterProcess(Process proc)
  {
    log.debug("unregisterProcess: " + proc);
    procs.remove(proc.getID());
  }

  private String getNamespaceURI(InputStream inStream)
  {
    DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
    dbf.setNamespaceAware(true);
    Document doc;
    try
    {
      DocumentBuilder db = dbf.newDocumentBuilder();
      doc = db.parse(inStream);
    }
    catch (Exception ex)
    {
      throw new IllegalStateException("Cannot parse process descriptor", ex);
    }

    Element root = doc.getDocumentElement();
    String nsURI = root.getNamespaceURI();
    if (nsURI == null)
      throw new IllegalStateException("Cannot get namespace URI from root element");

    return nsURI;
  }

  /**
   * Get the handler for the dialect with the given namespace URI
   */
  public DialectHandler getDialectHandler(String nsURI)
  {
    String dialectId = dialectRegistry.getDialect(nsURI);
    if (dialectId == null)
      throw new IllegalStateException("No dialect registered for: " + nsURI);

    DialectHandler dialectHandler = dialectHandlers.get(dialectId);
    if (dialectHandler == null)
      throw new IllegalStateException("No dialect handler registered for: " + dialectId);

    return dialectHandler;
  }
}