/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.bpm.model;

//$Id$

import java.util.List;

import javax.management.ObjectName;

import org.jboss.bpm.runtime.Attachments;

/**
 * A Process is any Activity performed within a company or organization.
 * 
 * @author thomas.diesler@jboss.com
 * @since 08-Jul-2008
 */
public interface Process extends SupportingElement, PropertySupport
{
  /**
   * Defines the type of a {@link Process}
   */
  public enum ProcessType
  {
    None, Private, Abstract, Collaboration
  }

  /**
   * Defines the status a {@link Process} can be in
   */
  public enum ProcessStatus
  {
    None, Ready, Active, Cancelled, Aborting, Aborted, Completing, Completed
  }

  /**
   * Get the unique name.
   */
  String getName();
  
  /**
   * Get the process state
   */
  ProcessStatus getProcessStatus();
  
  /**
   * Get the process type
   */
  ProcessType getProcessType();

  /**
   * Get the list of flow objects
   */
  List<FlowObject> getFlowObjects();

  /**
   * Get a list of flow objects of a given type.
   */
  <T extends FlowObject> List<T> getFlowObjects(Class<T> clazz);

  /**
   * Get a flow object by name.
   * @return null if not found
   */
  FlowObject getFlowObject(String name);
  
  /**
   * One or more Performers MAY be entered. The Performers attribute defines the resource that will be responsible for
   * the  The Performers entry could be in the form of a specific individual, a group, an organization role or
   * position, or an organization.
   */
  List<String> getPerformers();

  /**
   * One or more assignment expressions MAY be made for the object. The Assignment SHALL be performed as defined by the
   * AssignTime attribute.
   */
  List<Assignment> getAssignments();

  /**
   * The InputSets attribute defines the data requirements for input to the  Zero or more InputSets MAY be
   * defined. Each Input set is sufficient to allow the Process to be performed (if it has first been instantiated by
   * the appropriate signal arriving from an incoming Sequence Flow)
   */
  List<InputSet> getInputSets();

  /**
   * The OutputSets attribute defines the data requirements for output from the  Zero or more OutputSets MAY be
   * defined. At the completion of the Process, only one of the OutputSets may be produced--It is up to the
   * implementation of the Process to determine which set will be produced. However, the IORules attribute MAY indicate
   * a relationship between an OutputSet and an InputSet that started the 
   */
  List<OutputSet> getOutputSets();

  /**
   * Get the list of associated {@link Message} objects.
   * @return An empty list if there are none
   */
  List<Message> getMessages();

  /**
   * Get an associated {@link Message} by name.
   * @return null if not found
   */
  Message getMessage(String msgName);

  // Runtime Aspects ==================================================================================================== 
  
  /**
   * Start the process
   */
  ObjectName startProcess();

  /**
   * Start the process, with a given execution context
   */
  ObjectName startProcess(Attachments att);

  /**
   * All Tokens that are generated at the Start Event for that Process must eventually arrive at an End Event. 
   * The Process will be in a running state until all Tokens are consumed.
   * <p/>
   * This method until the process ends without timeout.
   */
  ProcessStatus waitForEnd();
  
  /**
   * All Tokens that are generated at the Start Event for that Process must eventually arrive at an End Event. 
   * The Process will be in a running state until all Tokens are consumed.
   * <p/>
   * This method until the process ends with a given timeout.
   */
  ProcessStatus waitForEnd(long timeout);
}