/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.bpm.cts.gateway.inclusive;

// $Id: InclusiveGatewayMergeTest.java 1987 2008-08-22 18:56:09Z thomas.diesler@jboss.com $

import java.io.IOException;
import java.util.List;

import org.jboss.bpm.client.SignalListener;
import org.jboss.bpm.client.SignalManager;
import org.jboss.bpm.model.EventBuilder;
import org.jboss.bpm.model.EventDetail;
import org.jboss.bpm.model.Process;
import org.jboss.bpm.model.ProcessBuilder;
import org.jboss.bpm.model.ProcessBuilderFactory;
import org.jboss.bpm.model.Signal;
import org.jboss.bpm.model.Gateway.GatewayType;
import org.jboss.bpm.model.Signal.SignalType;
import org.jboss.bpm.test.DefaultEngineTestCase;

/**
 * Inclusive gateway that has multiple incoming sequence flows. All tokens arriving from incoming sequence flows
 * proceeds unconditionally along the outgoing sequence flow. The inclusive join is stateless.
 * 
 * @author thomas.diesler@jboss.com
 * @since 06-Aug-2008
 */
public class InclusiveGatewayMergeTest extends DefaultEngineTestCase
{
  public void testGateA() throws Exception
  {
    Process proc = getProcess();

    // Add a signal listener that sends the other start trigger signal
    SignalManager sigManager = SignalManager.locateSignalManager();
    MergeListener sigListener = new MergeListener(new Signal(getTestID(), SignalType.SYSTEM_START_TRIGGER, "B"));
    sigManager.addSignalListener(sigListener);

    try
    {
      // Send start trigger signals
      proc.startProcess();
      sigManager.throwSignal(new Signal(getTestID(), SignalType.SYSTEM_START_TRIGGER, "A"));

      // Wait for the process to end
      proc.waitForEnd();
    }
    finally
    {
      sigManager.removeSignalListener(sigListener);
    }

    List<Signal> endSignals = getSignals(Signal.SignalType.SYSTEM_END_EVENT_EXIT);
    assertEquals(2, endSignals.size());
  }

  public Process getProcess() throws IOException
  {
    ProcessBuilder procBuilder = ProcessBuilderFactory.newInstance().newProcessBuilder();
    EventBuilder eventBuilder = procBuilder.addProcess(getName()).addStartEvent("StartA");
    eventBuilder.addEventDetail(EventDetail.EventDetailType.Signal).addSignalRef(SignalType.SYSTEM_START_TRIGGER, "A");
    eventBuilder = procBuilder.addSequenceFlow("Merge").addStartEvent("StartB");
    eventBuilder.addEventDetail(EventDetail.EventDetailType.Signal).addSignalRef(SignalType.SYSTEM_START_TRIGGER, "B");
    procBuilder.addSequenceFlow("Merge").addGateway("Merge", GatewayType.Inclusive).addSequenceFlow("End");
    procBuilder.addEndEvent("End");
    Process proc = procBuilder.getProcess();
    return proc;
  }

  class MergeListener implements SignalListener
  {
    private Signal nextSignal;

    public MergeListener(Signal nextSignal)
    {
      this.nextSignal = nextSignal;
    }

    public boolean acceptSignal(Signal signal)
    {
      return signal.getSignalType() == SignalType.SYSTEM_GATEWAY_ENTER;
    }

    public void catchSignal(Signal signal)
    {
      if (nextSignal != null)
      {
        SignalManager signalManager = SignalManager.locateSignalManager();
        signalManager.throwSignal(nextSignal);
        nextSignal = null;
      }
    }
  }
}
