/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.bpm.cts.task;

// $Id: ReceiveTaskTest.java 1987 2008-08-22 18:56:09Z thomas.diesler@jboss.com $

import java.io.IOException;

import org.jboss.bpm.InvalidProcessException;
import org.jboss.bpm.client.MessageManager;
import org.jboss.bpm.client.ProcessManager;
import org.jboss.bpm.client.SignalListener;
import org.jboss.bpm.client.SignalManager;
import org.jboss.bpm.model.EventBuilder;
import org.jboss.bpm.model.Message;
import org.jboss.bpm.model.MessageBuilder;
import org.jboss.bpm.model.MessageBuilderFactory;
import org.jboss.bpm.model.Process;
import org.jboss.bpm.model.ProcessBuilder;
import org.jboss.bpm.model.ProcessBuilderFactory;
import org.jboss.bpm.model.Signal;
import org.jboss.bpm.model.EventDetail.EventDetailType;
import org.jboss.bpm.model.Signal.SignalType;
import org.jboss.bpm.model.Task.TaskType;
import org.jboss.bpm.test.DefaultEngineTestCase;

/**
 * Test Receive Task
 * 
 * @author thomas.diesler@jboss.com
 * @since 03-Jul-2008
 */
public class ReceiveTaskTest extends DefaultEngineTestCase
{
  public void testReceiveTaskWithNoMessage() throws Exception
  {
    ProcessBuilder procBuilder = ProcessBuilderFactory.newInstance().newProcessBuilder();
    procBuilder.addProcess("ReceiveTaskTest").addStartEvent("Start").addSequenceFlow("TaskA");
    procBuilder.addTask("TaskA", TaskType.Receive).addSequenceFlow("End").addEndEvent("End");
    try
    {
      procBuilder.getProcess();
      fail("A Message for the MessageRef attribute MUST be entered");
    }
    catch (InvalidProcessException ex)
    {
      // expected
    }
  }

  public void testUnregisteredProcess() throws Exception
  {
    Process proc = getProcess();

    MessageManager msgManager = MessageManager.locateMessageManager();
    try
    {
      msgManager.sendMessage(proc.getID(), "TaskA", getMessage());
      fail("Send to an unregistered process is expected to fail");
    }
    catch (RuntimeException ex)
    {
      // expected
    }
  }

  public void testSuspendedMessage() throws Exception
  {
    Process proc = getProcess();
    ProcessManager procManager = ProcessManager.locateProcessManager();
    procManager.registerProcess(proc);

    // Send the message before the process is started
    MessageManager msgManager = MessageManager.locateMessageManager();
    msgManager.sendMessage(proc.getID(), "TaskA", getMessage());

    proc.startProcess();
    proc.waitForEnd();

    Message endMsg = getMessages().get(0);
    assertNotNull("End message expected", endMsg);
    assertEquals("bar", endMsg.getPropertyValue("foo"));
  }

  public void testSuspendedToken() throws Exception
  {
    final Process proc = getProcess();
    SignalListener sigListener = new SignalListener()
    {
      private boolean sendMessage = true;

      public boolean acceptSignal(Signal signal)
      {
        return signal.getSignalType() == SignalType.SYSTEM_TASK_EXIT;
      }

      public void catchSignal(Signal signal)
      {
        // Send the message after the process reached the receive task
        if (sendMessage == true)
        {
          sendMessage = false;
          MessageManager msgManager = MessageManager.locateMessageManager();
          msgManager.sendMessage(proc.getID(), "TaskA", getMessage());
        }
      }
    };
    SignalManager sigManager = SignalManager.locateSignalManager();
    sigManager.addSignalListener(sigListener);

    try
    {
      proc.startProcess();
      proc.waitForEnd();
    }
    finally
    {
      sigManager.removeSignalListener(sigListener);
    }

    Message endMsg = getMessages().get(0);
    assertNotNull("End message expected", endMsg);
    assertEquals("bar", endMsg.getPropertyValue("foo"));
  }

  private Message getMessage()
  {
    MessageBuilder procBuilder = MessageBuilderFactory.newInstance().newMessageBuilder();
    Message msg = procBuilder.newMessage("ReceiveTaskMessage").addProperty("foo", "bar", true).getMessage();
    return msg;
  }

  protected Process getProcess() throws IOException
  {
    ProcessBuilder procBuilder = ProcessBuilderFactory.newInstance().newProcessBuilder();
    MessageBuilder msgBuilder = procBuilder.addProcess("ReceiveTaskTest").addMessage("ReceiveTaskMessage");
    msgBuilder.addProperty("foo", null, true);
    msgBuilder = procBuilder.addMessage("EndEventMessage");
    msgBuilder.addToRef(getTestID()).addProperty("foo", null, true);
    procBuilder.addStartEvent("Start").addSequenceFlow("TaskA");
    procBuilder.addTask("TaskA", TaskType.Receive).addMessageRef("ReceiveTaskMessage");
    EventBuilder eventBuilder = procBuilder.addSequenceFlow("End").addEndEvent("End");
    eventBuilder.addEventDetail(EventDetailType.Message).addMessageRef("EndEventMessage");
    return procBuilder.getProcess();
  }
}
