/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.bpm.dialect.api10;

// $Id: ProcessMarshaller.java 1983 2008-08-22 11:12:51Z thomas.diesler@jboss.com $

import java.io.IOException;
import java.io.Writer;

import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBException;
import javax.xml.bind.Marshaller;

import org.jboss.bpm.NotImplementedException;
import org.jboss.bpm.dialect.api10.model.JAXBAssignment;
import org.jboss.bpm.dialect.api10.model.JAXBComplexGateway;
import org.jboss.bpm.dialect.api10.model.JAXBEndEvent;
import org.jboss.bpm.dialect.api10.model.JAXBEvent;
import org.jboss.bpm.dialect.api10.model.JAXBExclusiveGateway;
import org.jboss.bpm.dialect.api10.model.JAXBExecutionHandler;
import org.jboss.bpm.dialect.api10.model.JAXBExpression;
import org.jboss.bpm.dialect.api10.model.JAXBFlow;
import org.jboss.bpm.dialect.api10.model.JAXBFlowHandler;
import org.jboss.bpm.dialect.api10.model.JAXBFlowObject;
import org.jboss.bpm.dialect.api10.model.JAXBGateway;
import org.jboss.bpm.dialect.api10.model.JAXBInclusiveGateway;
import org.jboss.bpm.dialect.api10.model.JAXBInputSet;
import org.jboss.bpm.dialect.api10.model.JAXBMessage;
import org.jboss.bpm.dialect.api10.model.JAXBMessageEventDetail;
import org.jboss.bpm.dialect.api10.model.JAXBMessageRef;
import org.jboss.bpm.dialect.api10.model.JAXBOutputSet;
import org.jboss.bpm.dialect.api10.model.JAXBParallelGateway;
import org.jboss.bpm.dialect.api10.model.JAXBProcess;
import org.jboss.bpm.dialect.api10.model.JAXBProperty;
import org.jboss.bpm.dialect.api10.model.JAXBSequenceFlow;
import org.jboss.bpm.dialect.api10.model.JAXBSignal;
import org.jboss.bpm.dialect.api10.model.JAXBSignalEventDetail;
import org.jboss.bpm.dialect.api10.model.JAXBSignalHandler;
import org.jboss.bpm.dialect.api10.model.JAXBStartEvent;
import org.jboss.bpm.dialect.api10.model.JAXBTask;
import org.jboss.bpm.dialect.api10.model.ObjectFactory;
import org.jboss.bpm.model.Assignment;
import org.jboss.bpm.model.ComplexGateway;
import org.jboss.bpm.model.ConnectingObject;
import org.jboss.bpm.model.EndEvent;
import org.jboss.bpm.model.Event;
import org.jboss.bpm.model.EventDetail;
import org.jboss.bpm.model.ExclusiveGateway;
import org.jboss.bpm.model.Expression;
import org.jboss.bpm.model.FlowObject;
import org.jboss.bpm.model.Gate;
import org.jboss.bpm.model.Gateway;
import org.jboss.bpm.model.InclusiveGateway;
import org.jboss.bpm.model.InputSet;
import org.jboss.bpm.model.IntermediateEvent;
import org.jboss.bpm.model.Message;
import org.jboss.bpm.model.MessageEventDetail;
import org.jboss.bpm.model.MessageFlow;
import org.jboss.bpm.model.OutputSet;
import org.jboss.bpm.model.ParallelGateway;
import org.jboss.bpm.model.Participant;
import org.jboss.bpm.model.Process;
import org.jboss.bpm.model.Property;
import org.jboss.bpm.model.ReceiveTask;
import org.jboss.bpm.model.SendTask;
import org.jboss.bpm.model.SequenceFlow;
import org.jboss.bpm.model.Signal;
import org.jboss.bpm.model.SignalEventDetail;
import org.jboss.bpm.model.StartEvent;
import org.jboss.bpm.model.Task;
import org.jboss.bpm.model.SequenceFlow.ConditionType;
import org.jboss.bpm.runtime.ExecutionHandler;
import org.jboss.bpm.runtime.FlowHandler;
import org.jboss.bpm.runtime.HandlerSupport;
import org.jboss.bpm.runtime.SignalHandler;

/**
 * A JAXB marshaller for a Process
 * 
 * @author thomas.diesler@jboss.com
 * @since 08-Jul-2008
 */
public class ProcessMarshaller
{
  public void marshallProcess(Process proc, Writer out) throws JAXBException, IOException
  {
    JAXBProcess jaxbProc = adaptProcess(proc);
    JAXBContext jaxbContext = JAXBContext.newInstance(ObjectFactory.class);
    Marshaller marshaller = jaxbContext.createMarshaller();
    marshaller.setProperty(Marshaller.JAXB_FORMATTED_OUTPUT, true);
    marshaller.setProperty(Marshaller.JAXB_FRAGMENT, true);

    marshaller.marshal(jaxbProc, out);
  }

  private JAXBProcess adaptProcess(Process proc)
  {
    JAXBProcess jaxbProc = new JAXBProcess();
    jaxbProc.setName(proc.getName());

    for (Message msg : proc.getMessages())
    {
      JAXBMessage jaxbMsg = adaptMessage(msg);
      jaxbProc.getMessages().add(jaxbMsg);
    }
    for (Property prop : proc.getProperties())
    {
      JAXBProperty jaxbProp = getJaxbProperty(prop);
      jaxbProc.getProperties().add(jaxbProp);
    }
    for (Assignment ass : proc.getAssignments())
    {
      JAXBAssignment jaxbAss = getJaxbAssignment(ass);
      jaxbProc.getAssignments().add(jaxbAss);
    }

    for (FlowObject flowObject : proc.getFlowObjects())
    {
      JAXBFlowObject jaxbFlowObject;
      if (flowObject instanceof Event)
      {
        jaxbFlowObject = adaptEvent(jaxbProc, (Event)flowObject);
      }
      else if (flowObject instanceof Task)
      {
        jaxbFlowObject = adaptTask(jaxbProc, (Task)flowObject);
      }
      else if (flowObject instanceof Gateway)
      {
        jaxbFlowObject = adaptGateway(jaxbProc, (Gateway)flowObject);
      }
      else
      {
        throw new IllegalStateException("Unsupported flow object: " + flowObject);
      }
      
      for (Assignment ass : flowObject.getAssignments())
      {
        JAXBAssignment jaxbAss = getJaxbAssignment(ass);
        jaxbFlowObject.getAssignments().add(jaxbAss);
      }
      
      jaxbProc.getFlowObjects().add(jaxbFlowObject);
    }
    return jaxbProc;
  }


  private JAXBFlowObject adaptEvent(JAXBProcess jaxbProc, Event event)
  {
    JAXBEvent jaxbEvent;
    if (event instanceof StartEvent)
    {
      StartEvent start = (StartEvent)event;
      JAXBStartEvent jaxbStart = new JAXBStartEvent();
      jaxbStart.setName(start.getName());
      jaxbStart.setOutFlow(getJAXBFlow(start.getOutFlow()));

      if (start.getTrigger().size() > 1)
        throw new NotImplementedException("JBPM-1660", "StartTrigger Multiple");

      for (EventDetail trigger : start.getTrigger())
      {
        EventDetail.EventDetailType detailType = trigger.getEventDetailType();
        if (detailType == EventDetail.EventDetailType.Message)
        {
          throw new NotImplementedException("JBPM-1657", "StartTrigger Message");
        }
        else if (detailType == EventDetail.EventDetailType.Timer)
        {
          throw new NotImplementedException("JBPM-1658", "StartTrigger Timer");
        }
        else if (detailType == EventDetail.EventDetailType.Conditional)
        {
          throw new NotImplementedException("JBPM-1659", "StartTrigger Conditional");
        }
        else if (detailType == EventDetail.EventDetailType.Signal)
        {
          SignalEventDetail signalTrigger = (SignalEventDetail)trigger;
          Signal signal = signalTrigger.getSignalRef();
          JAXBSignalEventDetail jaxbTrigger = new JAXBSignalEventDetail();
          jaxbTrigger.setSignal(new JAXBSignal(signal.getSignalType(), signal.getMessage()));
          jaxbStart.getTrigger().add(jaxbTrigger);
        }
        else
        {
          throw new IllegalStateException("Unsupported start trigger: " + detailType);
        }
      }
      jaxbEvent = jaxbStart;
    }
    else if (event instanceof IntermediateEvent)
    {
      throw new NotImplementedException("JBPM-1661", "IntermediateEvent");
    }
    else if (event instanceof EndEvent)
    {
      EndEvent end = (EndEvent)event;
      JAXBEndEvent jaxbEnd = new JAXBEndEvent();
      jaxbEnd.setName(end.getName());

      if (end.getResult().size() > 1)
        throw new NotImplementedException("JBPM-1683", "EndEvent Multiple Result");

      for (EventDetail result : end.getResult())
      {
        EventDetail.EventDetailType detailType = result.getEventDetailType();
        if (detailType == EventDetail.EventDetailType.Message)
        {
          MessageEventDetail msgEventDetail = (MessageEventDetail)result;
          Message message = msgEventDetail.getMessageRef();
          JAXBMessageRef jaxbMessageRef = adaptMessageRef(jaxbProc, message);
          JAXBMessageEventDetail jaxbResult = new JAXBMessageEventDetail();
          jaxbResult.setMessageRef(jaxbMessageRef);
          jaxbEnd.getResult().add(jaxbResult);
        }
        else if (detailType == EventDetail.EventDetailType.Error)
        {
          throw new NotImplementedException("JBPM-1677", "EndEvent Error Result");
        }
        else if (detailType == EventDetail.EventDetailType.Cancel)
        {
          throw new NotImplementedException("JBPM-1678", "EndEvent Cancel Result");
        }
        else if (detailType == EventDetail.EventDetailType.Compensation)
        {
          throw new NotImplementedException("JBPM-1679", "EndEvent Compensation Result");
        }
        else if (detailType == EventDetail.EventDetailType.Signal)
        {
          throw new NotImplementedException("JBPM-1651", "EndEvent Signal Result");
        }
        else if (detailType == EventDetail.EventDetailType.Terminate)
        {
          throw new NotImplementedException("JBPM-1680", "EndEvent Terminate Result");
        }
        else
        {
          throw new IllegalStateException("Unsupported end event result type: " + detailType);
        }
      }

      jaxbEvent = jaxbEnd;
    }
    else
    {
      throw new IllegalStateException("Unsupported Event: " + event);
    }
    if (event instanceof HandlerSupport)
    {
      HandlerSupport hs = (HandlerSupport)event;
      jaxbEvent.setExecutionHandler(getJAXBExecutionHandler(event, hs.getExecutionHandler()));
      jaxbEvent.setFlowHandler(getJAXBFlowHandler(event, hs.getFlowHandler()));
      jaxbEvent.setSignalHandler(getJAXBSignalHandler(event, hs.getSignalHandler()));
    }
    return jaxbEvent;
  }

  private JAXBFlowObject adaptTask(JAXBProcess jaxbProc, Task task)
  {
    JAXBTask jaxbTask = new JAXBTask();
    jaxbTask.setTaskType(task.getTaskType());
    jaxbTask.setName(task.getName());
    jaxbTask.setOutFlow(getJAXBFlow(task.getOutFlow()));
    if (task instanceof HandlerSupport)
    {
      HandlerSupport hs = (HandlerSupport)task;
      jaxbTask.setExecutionHandler(getJAXBExecutionHandler(task, hs.getExecutionHandler()));
      jaxbTask.setFlowHandler(getJAXBFlowHandler(task, hs.getFlowHandler()));
      jaxbTask.setSignalHandler(getJAXBSignalHandler(task, hs.getSignalHandler()));
    }
    for (InputSet inSet : task.getInputSets())
    {
      JAXBInputSet jaxbSet = new JAXBInputSet();
      for (Property prop : inSet.getProperties())
      {
        JAXBProperty jaxbProp = getJaxbProperty(prop);
        jaxbSet.getProperties().add(jaxbProp);
      }
      jaxbTask.getInputSets().add(jaxbSet);
    }
    for (OutputSet outSet : task.getOutputSets())
    {
      JAXBOutputSet jaxbSet = new JAXBOutputSet();
      for (Property prop : outSet.getProperties())
      {
        JAXBProperty jaxbProp = getJaxbProperty(prop);
        jaxbSet.getProperties().add(jaxbProp);
      }
      jaxbTask.getOutputSets().add(jaxbSet);
    }
    for (Property prop : task.getProperties())
    {
      JAXBProperty jaxbProp = getJaxbProperty(prop);
      jaxbTask.getProperties().add(jaxbProp);
    }
    if (task instanceof ReceiveTask)
    {
      ReceiveTask recTask = (ReceiveTask)task;
      Message messageRef = recTask.getMessageRef();
      if (messageRef != null)
      {
        JAXBMessageRef jaxbMessageRef = adaptMessageRef(jaxbProc, messageRef);
        jaxbTask.setMessageRef(jaxbMessageRef);
      }
    }
    else if (task instanceof SendTask)
    {
      SendTask sendTask = (SendTask)task;
      Message messageRef = sendTask.getMessageRef();
      if (messageRef != null)
      {
        JAXBMessageRef jaxbMessageRef = adaptMessageRef(jaxbProc, messageRef);
        jaxbTask.setMessageRef(jaxbMessageRef);
      }
    }
    return jaxbTask;
  }

  private JAXBFlowObject adaptGateway(JAXBProcess jaxbProc, Gateway gateway)
  {
    JAXBGateway jaxbGateway;
    if (gateway instanceof ExclusiveGateway)
    {
      jaxbGateway = new JAXBExclusiveGateway();
    }
    else if (gateway instanceof InclusiveGateway)
    {
      jaxbGateway = new JAXBInclusiveGateway();
    }
    else if (gateway instanceof ParallelGateway)
    {
      jaxbGateway = new JAXBParallelGateway();
    }
    else if (gateway instanceof ComplexGateway)
    {
      jaxbGateway = new JAXBComplexGateway();
    }
    else
    {
      throw new IllegalStateException("Unsupported gateway: " + gateway);
    }
    jaxbGateway.setName(gateway.getName());
    if (gateway instanceof HandlerSupport)
    {
      HandlerSupport hs = (HandlerSupport)gateway;
      jaxbGateway.setExecutionHandler(getJAXBExecutionHandler(gateway, hs.getExecutionHandler()));
      jaxbGateway.setFlowHandler(getJAXBFlowHandler(gateway, hs.getFlowHandler()));
      jaxbGateway.setSignalHandler(getJAXBSignalHandler(gateway, hs.getSignalHandler()));
    }
    for (Gate gate : gateway.getGates())
    {
      SequenceFlow flow = gate.getOutgoingSequenceFlow();
      jaxbGateway.getOutFlows().add(getJAXBFlow(flow));
    }
    return jaxbGateway;
  }

  private JAXBMessageRef adaptMessageRef(JAXBProcess jaxbProc, Message msgRef)
  {
    String msgName = msgRef.getName();
    JAXBMessage jaxbMsg = jaxbProc.getMessageByName(msgName);
    JAXBMessageRef jaxbMsgRef = new JAXBMessageRef();
    jaxbMsgRef.setNameRef(msgName);

    String fromStr = getParticipant(msgRef.getFromRef());
    if (fromStr != null && fromStr.equals(jaxbMsg.getFromRef()) == false)
      jaxbMsgRef.setFromRef(fromStr);

    String toStr = getParticipant(msgRef.getToRef());
    if (toStr != null && toStr.equals(jaxbMsg.getToRef()) == false)
      jaxbMsgRef.setToRef(toStr);

    return jaxbMsgRef;
  }

  private JAXBMessage adaptMessage(Message msg)
  {
    JAXBMessage jaxbMsg = new JAXBMessage();
    jaxbMsg.setName(msg.getName());
    jaxbMsg.setFromRef(getParticipant(msg.getFromRef()));
    jaxbMsg.setToRef(getParticipant(msg.getToRef()));
    for (Property prop : msg.getProperties())
    {
      JAXBProperty jaxbProp = getJaxbProperty(prop);
      jaxbMsg.getProperties().add(jaxbProp);
    }
    return jaxbMsg;
  }

  private String getParticipant(Participant par)
  {
    String parName = null;
    if (par != null)
    {
      if (par.getEntityRef() != null)
        parName = par.getEntityRef().getName().getCanonicalName();
      else if (par.getRoleRef() != null)
        parName = par.getRoleRef().getName().getCanonicalName();
    }
    return parName;
  }

  private JAXBFlow getJAXBFlow(ConnectingObject flow)
  {
    JAXBFlow jaxb;
    if (flow instanceof SequenceFlow)
    {
      SequenceFlow seqFlow = (SequenceFlow)flow;
      JAXBSequenceFlow jaxbSeq = new JAXBSequenceFlow();
      if (seqFlow.getConditionType() == ConditionType.Expression)
      {
        jaxbSeq.setConditionType(seqFlow.getConditionType());
        jaxbSeq.setCondition(getJAXBExpression(seqFlow.getConditionExpression()));
      }
      else if (seqFlow.getConditionType() == ConditionType.Default)
      {
        jaxbSeq.setConditionType(seqFlow.getConditionType());
      }
      jaxb = jaxbSeq;
    }
    else if (flow instanceof MessageFlow)
    {
      throw new NotImplementedException("JBPM-1382", "Message Flow");
    }
    else
    {
      throw new IllegalStateException("Unsupported connectiong object: " + flow);
    }
    jaxb.setName(flow.getName());
    jaxb.setTargetName(flow.getTargetName());
    return jaxb;
  }

  private JAXBProperty getJaxbProperty(Property prop)
  {
    JAXBProperty jaxbProp = new JAXBProperty();
    jaxbProp.setName(prop.getName());
    jaxbProp.setValue((String)prop.getValue());
    if (prop.isCorrelation())
      jaxbProp.setCorrelation(prop.isCorrelation());
    return jaxbProp;
  }

  private JAXBExpression getJAXBExpression(Expression expr)
  {
    JAXBExpression jaxb = new JAXBExpression();
    jaxb.setLang(expr.getExpressionLanguage());
    jaxb.setBody(expr.getExpressionBody());
    return jaxb;
  }

  private JAXBAssignment getJaxbAssignment(Assignment ass)
  {
    JAXBAssignment jaxbAss = new JAXBAssignment();
    jaxbAss.setAssignTime(ass.getAssignTime());
    jaxbAss.setFrom(getJAXBExpression(ass.getFrom()));
    jaxbAss.setTo(ass.getTo().getName());
    return jaxbAss;
  }
  
  private JAXBExecutionHandler getJAXBExecutionHandler(FlowObject fo, ExecutionHandler handler)
  {
    JAXBExecutionHandler jaxb = null;
    if (handler != null)
    {
      if (fo.getClass().getPackage() != handler.getClass().getPackage())
      {
        jaxb = new JAXBExecutionHandler();
        jaxb.setClassName(handler.getClass().getName());
      }
    }
    return jaxb;
  }

  private JAXBFlowHandler getJAXBFlowHandler(FlowObject fo, FlowHandler handler)
  {
    JAXBFlowHandler jaxb = null;
    if (handler != null)
    {
      if (fo.getClass().getPackage() != handler.getClass().getPackage())
      {
        jaxb = new JAXBFlowHandler();
        jaxb.setClassName(handler.getClass().getName());
      }
    }
    return jaxb;
  }

  private JAXBSignalHandler getJAXBSignalHandler(FlowObject fo, SignalHandler handler)
  {
    JAXBSignalHandler jaxb = null;
    if (handler != null)
    {
      if (fo.getClass().getPackage() != handler.getClass().getPackage())
      {
        jaxb = new JAXBSignalHandler();
        jaxb.setClassName(handler.getClass().getName());
      }
    }
    return jaxb;
  }
}
