/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.bpm.dialect.stp;

// $Id: ProcessUnmarshaller.java 1930 2008-08-19 11:04:43Z thomas.diesler@jboss.com $

import java.io.InputStream;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBException;
import javax.xml.bind.Unmarshaller;
import javax.xml.namespace.QName;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;

import org.jboss.bpm.InvalidProcessException;
import org.jboss.bpm.dialect.stp.model.Activity;
import org.jboss.bpm.dialect.stp.model.ActivityType;
import org.jboss.bpm.dialect.stp.model.BpmnDiagram;
import org.jboss.bpm.dialect.stp.model.ObjectFactory;
import org.jboss.bpm.dialect.stp.model.Pool;
import org.jboss.bpm.dialect.stp.model.SequenceEdge;
import org.jboss.bpm.model.Gateway;
import org.jboss.bpm.model.Process;
import org.jboss.bpm.model.ProcessBuilder;
import org.jboss.bpm.model.ProcessBuilderFactory;
import org.jboss.bpm.model.Task;
import org.jboss.util.xml.DOMUtils;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

/**
 * A JAXB unmarshaller for a Process
 * 
 * @author thomas.diesler@jboss.com
 * @since 08-Jul-2008
 */
public class ProcessUnmarshaller
{
  public static final String NAMESPACE_URI = "http://stp.eclipse.org/bpmn";

  private BpmnDiagram diagram;
  private List<SequenceEdge> sequenceEdges;
  private List<Activity> activities = new ArrayList<Activity>();

  @SuppressWarnings("unchecked")
  public Process unmarshallProcess(InputStream xml, boolean isInclude) throws JAXBException
  {
    JAXBContext jaxbContext = JAXBContext.newInstance(ObjectFactory.class);
    Unmarshaller unmarshaller = jaxbContext.createUnmarshaller();
    unmarshaller.setProperty("com.sun.xml.bind.ObjectFactory", new ObjectFactory());

    // We need to unmarshall the elements one by one because the generated STP BPMN
    // file does not contain xsd:type declarations so that JAXB cannot createt the types
    // automatically
    DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
    dbf.setNamespaceAware(true);
    Document doc;
    try
    {
      DocumentBuilder db = dbf.newDocumentBuilder();
      doc = db.parse(xml);
    }
    catch (Exception ex)
    {
      throw new IllegalStateException("Cannot parse stpbpmn", ex);
    }

    Element root = doc.getDocumentElement();
    diagram = unmarshaller.unmarshal(root, BpmnDiagram.class).getValue();
    ProcessBuilder procBuilder = adaptDiagram(diagram);

    // Initialize the list of sequence edges
    Element poolEl = (Element)DOMUtils.getChildElements(root, "pools").next();
    Pool pool = unmarshaller.unmarshal(poolEl, Pool.class).getValue();
    sequenceEdges = pool.getSequenceEdges();

    // Iterate over all 'verticies'
    Iterator<Element> itEl = DOMUtils.getChildElements(poolEl, "vertices");
    while (itEl.hasNext())
    {
      Element stpEl = itEl.next();
      String xmiType = DOMUtils.getAttributeValue(stpEl, new QName("http://www.omg.org/XMI", "type"));
      if ("bpmn:Activity".equals(xmiType))
      {
        Activity stpActivity = unmarshaller.unmarshal(stpEl, Activity.class).getValue();
        String name = DOMUtils.getAttributeValue(stpEl, "name");
        name = (name != null ? name : stpActivity.getId());
        stpActivity.setLabel(name);

        // Fall back to the attribute if JAXB did not initialize it
        List<String> outEdges = stpActivity.getOutgoingEdges();
        if (outEdges.size() == 0)
        {
          String edgeStr = DOMUtils.getAttributeValue(stpEl, "outgoingEdges");
          if (edgeStr != null && edgeStr.length() > 0)
            outEdges.add(edgeStr);
        }

        activities.add(stpActivity);
      }
      else
      {
        throw new IllegalStateException("Unsupported xmi:type: " + xmiType);
      }
    }

    // Build the activities
    for (Activity stpActivity : activities)
    {
      adaptActivity(procBuilder, stpActivity);
    }

    Process proc = (isInclude ? procBuilder.getProcessForInclude() : procBuilder.getProcess());
    return proc;
  }

  private ProcessBuilder adaptDiagram(BpmnDiagram bpmnDiagram)
  {
    List<Pool> stpPools = bpmnDiagram.getPools();
    if (stpPools == null || stpPools.size() == 0)
      throw new IllegalStateException("Expected at least one Pool");
    if (stpPools.size() > 1)
      throw new IllegalStateException("Multiple Pools not supported");

    Pool stpPool = stpPools.get(0);
    ProcessBuilder builder = ProcessBuilderFactory.newInstance().newProcessBuilder();
    builder.addProcess(stpPool.getId());

    return builder;
  }

  private void adaptActivity(ProcessBuilder builder, Activity stpActivity)
  {
    ActivityType activityType = stpActivity.getActivityType();
    if (activityType == ActivityType.EVENT_START_EMPTY)
    {
      builder.addStartEvent("Start");
      adaptOutgoingEdges(builder, stpActivity);
    }
    else if (activityType == ActivityType.EVENT_END_EMPTY)
    {
      String name = stpActivity.getLabel();
      builder.addEndEvent(name);
    }
    else if (activityType == ActivityType.TASK)
    {
      String name = stpActivity.getLabel();
      
      Task.TaskType taskType = Task.TaskType.None;
      if (name.startsWith("Send"))
        taskType = Task.TaskType.Send;
      if (name.startsWith("Receive"))
        taskType = Task.TaskType.Receive;
      
      builder.addTask(name, taskType);
      adaptOutgoingEdges(builder, stpActivity);
    }
    else if (activityType == ActivityType.GATEWAY_DATA_BASED_EXCLUSIVE)
    {
      String name = stpActivity.getLabel();
      builder.addGateway(name, Gateway.GatewayType.Exclusive);
      adaptOutgoingEdges(builder, stpActivity);
    }
    else if (activityType == ActivityType.GATEWAY_PARALLEL)
    {
      String name = stpActivity.getLabel();
      builder.addGateway(name, Gateway.GatewayType.Parallel);
      adaptOutgoingEdges(builder, stpActivity);
    }
    else
    {
      throw new InvalidProcessException("Unsupported activity type: " + activityType);
    }
  }

  private void adaptOutgoingEdges(ProcessBuilder builder, Activity stpActivity)
  {
    for (String edgeIds : stpActivity.getOutgoingEdges())
    {
      String[] edgeIdArr = edgeIds.split("\\s");
      for (String edgeId : edgeIdArr)
      {
        SequenceEdge seqEdge = getSequenceEdgeById(edgeId);
        String targetId = seqEdge.getTarget();
        Activity targetAct = getActivityById(targetId);
        String targetName = targetAct.getLabel();
        builder.addSequenceFlow(targetName);
      }
    }
  }

  private SequenceEdge getSequenceEdgeById(String edgeId)
  {
    for (SequenceEdge seqEdge : sequenceEdges)
    {
      if (seqEdge.getId().equals(edgeId))
        return seqEdge;
    }
    throw new InvalidProcessException("Cannot find SequenceEdge: " + edgeId);
  }

  private Activity getActivityById(String id)
  {
    for (Activity act : activities)
    {
      if (act.getId().equals(id))
        return act;
    }
    throw new InvalidProcessException("Cannot find Activity: " + id);
  }
}
