/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.bpm.samples.airticket;

// $Id: AirticketProcessBuilder.java 1989 2008-08-22 20:13:51Z thomas.diesler@jboss.com $

import javax.management.ObjectName;

import org.jboss.bpm.model.Expression;
import org.jboss.bpm.model.Gateway;
import org.jboss.bpm.model.GatewayBuilder;
import org.jboss.bpm.model.Process;
import org.jboss.bpm.model.ProcessBuilder;
import org.jboss.bpm.model.ProcessBuilderFactory;
import org.jboss.bpm.model.Task;
import org.jboss.bpm.model.TaskBuilder;
import org.jboss.bpm.model.Assignment.AssignTime;
import org.jboss.bpm.model.Expression.ExpressionLanguage;

/**
 * Test the Airticket sample application
 * 
 * @author thomas.diesler@jboss.com
 * @since 03-Jul-2008
 */
public class AirticketProcessBuilder
{
  public static final String PROCESS_NAME = "Airticket";
  
  public static final String EVENT_END = "End";

  public static final String GATEWAY_IS_ACCEPTED = "IsAccepted";
  public static final String GATEWAY_IS_AVAILABLE = "IsAvailable";
  public static final String GATEWAY_IS_VALID = "IsValid";
  public static final String GATEWAY_SPLIT = "Split";
  public static final String GATEWAY_SYNC_JOIN = "SyncJoin";
  
  public static final String TASK_BOOKING = "Booking";
  public static final String TASK_CHECK_AVAILABLE = "CheckAvailable";
  public static final String TASK_CREDIT_CARD = "CreditCard";
  public static final String TASK_RECEIVE_ORDER = "ReceiveOrder";
  public static final String TASK_RECEIVE_REQUEST = "ReceiveReq";
  public static final String TASK_SEND_CONFIRM = "SendConfirm";
  public static final String TASK_SEND_NOT_AVAILABLE = "SendNotAvailable";
  public static final String TASK_SEND_OFFER = "SendOffer";
  public static final String TASK_SEND_INVALID = "SendInvalid";
  public static final String TASK_VALIDATE = "Validate";
  
  public static final String MESSAGE_CONFIRM = "ConfirmMsg";
  public static final String MESSAGE_ACCEPT_OFFER = "AcceptOfferMsg";
  public static final String MESSAGE_OFFER = "OfferMsg";
  public static final String MESSAGE_NOT_AVAILABLE = "NotAvailableMsg";
  public static final String MESSAGE_INVALID_DATA = "InvalidDataMsg";
  public static final String MESSAGE_REQ_DATA = "ReqDataMsg";
  
  public static final String PROPERTY_NAME = "Name";
  public static final String PROPERTY_FROM = "From";
  public static final String PROPERTY_TO = "To";
  public static final String PROPERTY_DATE = "Date";
  public static final String PROPERTY_SEATS = "Seats";
  public static final String PROPERTY_CREDIT_CARD = "CreditCard";
  public static final String PROPERTY_EXPIRE = "Expire";
  public static final String PROPERTY_PRICE = "Price";
  
  public static final String PROPERTY_IS_VALID_REQUEST = "isReqDataValid";
  public static final String PROPERTY_IS_AVAILABLE = "isAvailable";
  public static final String PROPERTY_IS_OFFER_ACCEPTED = "isOfferAccepted";
  
  private ObjectName msgListenerID;
  
  public AirticketProcessBuilder(ObjectName msgListenerID)
  {
    this.msgListenerID = msgListenerID;
  }

  public Process buildProcess()
  {
    // Create a Process through the ProcessBuilder
    ProcessBuilderFactory procFactory = ProcessBuilderFactory.newInstance();
    ProcessBuilder procBuilder = procFactory.newProcessBuilder();
    procBuilder.addProcess(PROCESS_NAME);
    
    // Add Start Event
    procBuilder.addStartEvent("Start").addSequenceFlow(TASK_RECEIVE_REQUEST);

    // Build the ReceiveReqTask
    TaskBuilder taskBuilder = procBuilder.addTask(TASK_RECEIVE_REQUEST, Task.TaskType.Receive);
    taskBuilder.addMessageRef(MESSAGE_REQ_DATA).
    addAssignment(AssignTime.End, ExpressionLanguage.MVEL, getValidationExpression(), PROPERTY_IS_VALID_REQUEST).
    addSequenceFlow(GATEWAY_IS_VALID);

    // Build the ValidateGateway
    GatewayBuilder gatewayBuilder = procBuilder.addGateway(GATEWAY_IS_VALID, Gateway.GatewayType.Exclusive);
    gatewayBuilder.addConditionalGate(TASK_CHECK_AVAILABLE, Expression.ExpressionLanguage.MVEL, PROPERTY_IS_VALID_REQUEST + " == true").
    addDefaultGate(TASK_SEND_INVALID);

    // Build the CheckAvailabilityTask
    taskBuilder = procBuilder.addTask(TASK_CHECK_AVAILABLE);
    taskBuilder.addAssignment(AssignTime.End, ExpressionLanguage.MVEL, PROPERTY_SEATS + " < 10", PROPERTY_IS_AVAILABLE).
    addAssignment(AssignTime.End, ExpressionLanguage.MVEL, PROPERTY_SEATS + " * 111", PROPERTY_PRICE).
    addSequenceFlow(GATEWAY_IS_AVAILABLE);

    // Build the SendInvalidTask
    taskBuilder = procBuilder.addTask(TASK_SEND_INVALID, Task.TaskType.Send);
    taskBuilder.addMessageRef(MESSAGE_INVALID_DATA).
    addSequenceFlow(TASK_RECEIVE_REQUEST);

    // Build the AvailabilityGateway
    gatewayBuilder = procBuilder.addGateway(GATEWAY_IS_AVAILABLE, Gateway.GatewayType.Exclusive);
    gatewayBuilder.addConditionalGate(TASK_SEND_OFFER, Expression.ExpressionLanguage.MVEL, PROPERTY_IS_AVAILABLE + " == true").
    addDefaultGate(TASK_SEND_NOT_AVAILABLE);
    
    // Build the SendOfferTask
    taskBuilder = procBuilder.addTask(TASK_SEND_OFFER, Task.TaskType.Send);
    taskBuilder.addMessageRef(MESSAGE_OFFER).
    addSequenceFlow(TASK_RECEIVE_ORDER);

    // Build the SendNotAvailableTask
    taskBuilder = procBuilder.addTask(TASK_SEND_NOT_AVAILABLE, Task.TaskType.Send);
    taskBuilder.addMessageRef(MESSAGE_NOT_AVAILABLE).
    addSequenceFlow(TASK_RECEIVE_REQUEST);

    // Build the ReceiveReqTask
    taskBuilder = procBuilder.addTask(TASK_RECEIVE_ORDER, Task.TaskType.Receive);
    taskBuilder.addMessageRef(MESSAGE_ACCEPT_OFFER).
    addSequenceFlow(GATEWAY_IS_ACCEPTED);
    
    // Build the AcceptOfferGateway
    gatewayBuilder = procBuilder.addGateway(GATEWAY_IS_ACCEPTED, Gateway.GatewayType.Exclusive);
    gatewayBuilder.addConditionalGate(GATEWAY_SPLIT, Expression.ExpressionLanguage.MVEL, PROPERTY_IS_OFFER_ACCEPTED + " == true").
    addDefaultGate(TASK_RECEIVE_REQUEST);
    
    // Build the AcceptOfferGateway
    gatewayBuilder = procBuilder.addGateway(GATEWAY_SPLIT, Gateway.GatewayType.Parallel);
    gatewayBuilder.addGate(TASK_CREDIT_CARD).
    addGate(TASK_BOOKING);
    
    // Build the CreditCardTask
    taskBuilder = procBuilder.addTask(TASK_CREDIT_CARD);
    taskBuilder.addInputSet().
    addPropertyInput(PROPERTY_CREDIT_CARD).
    addPropertyInput(PROPERTY_EXPIRE).
    addPropertyInput(PROPERTY_PRICE).
    addSequenceFlow(GATEWAY_SYNC_JOIN);
    
    // Build the BookingTask
    taskBuilder = procBuilder.addTask(TASK_BOOKING);
    taskBuilder.addInputSet().
    addPropertyInput(PROPERTY_NAME).
    addPropertyInput(PROPERTY_FROM).
    addPropertyInput(PROPERTY_TO).
    addPropertyInput(PROPERTY_DATE).
    addPropertyInput(PROPERTY_SEATS).
    addSequenceFlow(GATEWAY_SYNC_JOIN);
    
    // Build the SynchronizedJoinGateway
    gatewayBuilder = procBuilder.addGateway(GATEWAY_SYNC_JOIN, Gateway.GatewayType.Parallel);
    gatewayBuilder.addGate(TASK_SEND_CONFIRM);
    
    // Build the SendConfirmTask
    taskBuilder = procBuilder.addTask(TASK_SEND_CONFIRM, Task.TaskType.Send);
    taskBuilder.addMessageRef(MESSAGE_CONFIRM).
    addSequenceFlow(EVENT_END);
    
    // Add the EndEvent
    procBuilder.addEndEvent(EVENT_END);

    // Add ReqDataMsg 
    procBuilder.addMessage(MESSAGE_REQ_DATA).
    addProperty(PROPERTY_NAME, null, true).
    addProperty(PROPERTY_FROM, null, true).
    addProperty(PROPERTY_TO, null, true).
    addProperty(PROPERTY_DATE, null, true).
    addProperty(PROPERTY_SEATS, null, true);
    
    // Add InvalidDataMsg 
    procBuilder.addMessage(MESSAGE_INVALID_DATA).
    addToRef(msgListenerID).
    addProperty(PROPERTY_NAME, null, true).
    addProperty(PROPERTY_FROM, null, true).
    addProperty(PROPERTY_TO, null, true).
    addProperty(PROPERTY_DATE, null, true).
    addProperty(PROPERTY_SEATS, null, true);
    
    // Add NotAvailableMsg 
    procBuilder.addMessage(MESSAGE_NOT_AVAILABLE).
    addToRef(msgListenerID).
    addProperty(PROPERTY_NAME, null, true).
    addProperty(PROPERTY_FROM, null, true).
    addProperty(PROPERTY_TO, null, true).
    addProperty(PROPERTY_DATE, null, true).
    addProperty(PROPERTY_SEATS, null, true);
    
    // Add OfferMsg 
    procBuilder.addMessage(MESSAGE_OFFER).
    addToRef(msgListenerID).
    addProperty(PROPERTY_NAME, null, true).
    addProperty(PROPERTY_FROM, null, true).
    addProperty(PROPERTY_TO, null, true).
    addProperty(PROPERTY_DATE, null, true).
    addProperty(PROPERTY_SEATS, null, true).
    addProperty(PROPERTY_PRICE, null, true);
    
    // Add AcceptOfferMsg 
    procBuilder.addMessage(MESSAGE_ACCEPT_OFFER).
    addProperty(TASK_CREDIT_CARD, null, true).
    addProperty(PROPERTY_EXPIRE, null, true).
    addProperty(PROPERTY_IS_OFFER_ACCEPTED, null, true);
    
    // Add ConfirmMsg 
    procBuilder.addMessage(MESSAGE_CONFIRM).
    addToRef(msgListenerID).
    addProperty(PROPERTY_NAME, null, true).
    addProperty(PROPERTY_FROM, null, true).
    addProperty(PROPERTY_TO, null, true).
    addProperty(PROPERTY_DATE, null, true).
    addProperty(PROPERTY_SEATS, null, true).
    addProperty(PROPERTY_PRICE, null, true);
    
    // Get the process
    Process proc = procBuilder.getProcess();
    return proc;
  }
  
  private String getValidationExpression()
  {
    StringBuilder str = new StringBuilder(PROPERTY_NAME + " != null && ");
    str.append(PROPERTY_FROM + " != null && ");
    str.append(PROPERTY_TO + " != null && ");
    str.append(PROPERTY_DATE + " != null && ");
    str.append(PROPERTY_SEATS + " > 0");
    return str.toString();
  }
}