/*
 * Decompiled with CFR 0.152.
 */
package org.kie.api.internal.utils;

import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStreamReader;
import java.net.JarURLConnection;
import java.net.URL;
import java.net.URLConnection;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.TreeMap;
import java.util.function.Consumer;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ServiceDiscoveryImpl {
    private static final String[] KIE_MODULES = new String[]{"", "drools-alphanetwork-compiler", "drools-beliefs", "drools-compiler", "drools-core", "drools-decisiontables", "drools-metric", "drools-model-compiler", "drools-mvel", "drools-persistence-jpa", "drools-scorecards", "drools-serialization-protobuf", "drools-traits", "drools-workbench-models-guided-dtable", "drools-workbench-models-guided-scorecard", "drools-workbench-models-guided-template", "drools-xml-support", "jbpm-bpmn2", "jbpm-case-mgmt-cmmn", "jbpm-flow", "jbpm-flow-builder", "jbpm-human-task-jpa", "kie-ci", "kie-dmn-core", "kie-dmn-jpmml", "kie-internal", "kie-pmml", "kie-pmml-evaluator-assembler", "kie-pmml-evaluator-core", "kie-server-services-jbpm-cluster"};
    private static final Logger log = LoggerFactory.getLogger(ServiceDiscoveryImpl.class);
    private static final String CONF_FILE_FOLDER = "META-INF/kie";
    private static final String CONF_FILE_NAME = "kie.conf";
    public static final String LEGACY_CONF_FILE = "META-INF/kie.conf";
    private final PriorityMap<String, Object> services = new PriorityMap();
    private final Map<String, List<?>> childServices = new HashMap();
    private Map<String, List<Object>> cachedServices;
    private boolean sealed = false;

    ServiceDiscoveryImpl() {
    }

    public static ServiceDiscoveryImpl getInstance() {
        return LazyHolder.INSTANCE;
    }

    public <T> void addService(Class<T> serviceClass, T service) {
        this.addService(serviceClass.getCanonicalName(), service);
    }

    public synchronized void addService(String serviceName, Object object) {
        if (this.sealed) {
            throw new IllegalStateException("Unable to add service '" + serviceName + "'. Services cannot be added once the ServiceDiscovery is sealed");
        }
        this.cachedServices.computeIfAbsent(serviceName, n -> new ArrayList()).add(object);
    }

    public synchronized void reset() {
        this.cachedServices = null;
        this.sealed = false;
        this.services.reset();
    }

    public synchronized Map<String, List<Object>> getServices() {
        if (!this.sealed) {
            this.getKieConfs().ifPresent(kieConfs -> {
                for (URL kieConfUrl : ((KieConfs)kieConfs).resources) {
                    this.registerConfs(((KieConfs)kieConfs).classLoader, kieConfUrl);
                }
            });
            this.cachedServices = Collections.unmodifiableMap(this.buildMap());
            this.sealed = true;
        }
        return this.cachedServices;
    }

    public void registerConfs(ClassLoader classLoader, URL url) {
        log.debug("Loading kie.conf from {} in classloader {}", (Object)url, (Object)classLoader);
        try (BufferedReader br = new BufferedReader(new InputStreamReader(url.openStream()));){
            String line = br.readLine();
            while (line != null) {
                if (line.contains("=") && !line.contains("[")) {
                    String[] entry = line.split("=");
                    this.processKieService(classLoader, entry[0].trim(), entry[1].trim());
                }
                line = br.readLine();
            }
        }
        catch (Exception e) {
            throw new RuntimeException("Unable to build kie service url = " + url.toExternalForm(), e);
        }
    }

    private void processKieService(ClassLoader classLoader, String key, String values) {
        for (String value : values.split(",")) {
            boolean optional = key.startsWith("?");
            String serviceName = optional ? key.substring(1) : key;
            try {
                if (value.startsWith("+")) {
                    this.childServices.computeIfAbsent(serviceName, k -> new ArrayList()).add(this.newInstance(classLoader, value.substring(1)));
                    log.debug("Added child Service {}", (Object)value);
                    continue;
                }
                String[] splitValues = value.split(";");
                if (splitValues.length > 2) {
                    throw new RuntimeException("Invalid kie.conf entry: " + value);
                }
                int priority = splitValues.length == 2 ? Integer.parseInt(splitValues[1].trim()) : 0;
                this.services.put(priority, serviceName, this.newInstance(classLoader, splitValues[0].trim()));
                log.debug("Added Service {} with priority {}", (Object)value, (Object)priority);
            }
            catch (RuntimeException e) {
                if (optional) {
                    log.info("Cannot load service: {}", (Object)serviceName);
                    continue;
                }
                log.error("Loading failed because {}", (Object)e.getMessage());
                throw e;
            }
        }
    }

    private <T> T newInstance(ClassLoader classLoader, String className) {
        try {
            return (T)Class.forName(className, true, classLoader).getConstructor(new Class[0]).newInstance(new Object[0]);
        }
        catch (Throwable t) {
            throw new RuntimeException("Cannot create instance of class: " + className, t);
        }
    }

    private Map<String, List<Object>> buildMap() {
        HashMap<String, List<Object>> servicesMap = new HashMap<String, List<Object>>();
        for (Map.Entry<String, List<Object>> entry : this.services.entrySet()) {
            log.debug("Service {} is implemented by {}", (Object)entry.getKey(), entry.getValue().get(0));
            servicesMap.put(entry.getKey(), entry.getValue());
            List<?> children = this.childServices.remove(entry.getKey());
            if (children == null) continue;
            for (Object child : children) {
                for (Object service : entry.getValue()) {
                    ((Consumer)service).accept(child);
                }
            }
        }
        if (!this.childServices.isEmpty()) {
            throw new RuntimeException("Child services " + this.childServices.keySet() + " have no parent");
        }
        if (log.isTraceEnabled()) {
            for (Map.Entry<String, List<Object>> entry : servicesMap.entrySet()) {
                if (entry.getValue().size() == 1) {
                    log.trace("Service {} is implemented by {}", (Object)entry.getKey(), entry.getValue().get(0));
                    continue;
                }
                log.trace("Service {} is implemented (in order of priority) by {}", (Object)entry.getKey(), entry.getValue());
            }
        }
        return servicesMap;
    }

    private Optional<KieConfs> getKieConfs() {
        return Stream.of(this.getClass().getClassLoader(), Thread.currentThread().getContextClassLoader(), ClassLoader.getSystemClassLoader()).map(this::loadKieConfs).filter(Objects::nonNull).findFirst();
    }

    private KieConfs loadKieConfs(ClassLoader cl) {
        if (cl == null) {
            return null;
        }
        try {
            Collection<URL> resources = ServiceDiscoveryImpl.findKieConfUrls(cl);
            return resources.isEmpty() ? null : new KieConfs(cl, resources);
        }
        catch (IOException e) {
            return null;
        }
    }

    private static Collection<URL> findKieConfUrls(ClassLoader cl) throws IOException {
        List<Object> kieConfsUrls = new ArrayList();
        Enumeration<URL> metaInfs = cl.getResources(CONF_FILE_FOLDER);
        while (metaInfs.hasMoreElements()) {
            URL metaInf = metaInfs.nextElement();
            if (metaInf.getProtocol().startsWith("vfs")) {
                kieConfsUrls.clear();
                break;
            }
            URLConnection con = metaInf.openConnection();
            if (con instanceof JarURLConnection) {
                ServiceDiscoveryImpl.collectKieConfsInJar(kieConfsUrls, metaInf, (JarURLConnection)con);
                continue;
            }
            ServiceDiscoveryImpl.collectKieConfsInFile(kieConfsUrls, new File(metaInf.getFile()));
        }
        if (kieConfsUrls.isEmpty()) {
            kieConfsUrls = ServiceDiscoveryImpl.getKieConfsFromKnownModules(cl).collect(Collectors.toList());
        } else {
            List notRegisteredModules = kieConfsUrls.stream().map(ServiceDiscoveryImpl::getModuleName).filter(module -> Arrays.binarySearch(KIE_MODULES, module) < 0).collect(Collectors.toList());
            if (!notRegisteredModules.isEmpty()) {
                throw new IllegalStateException("kie.conf file discovered for modules " + notRegisteredModules + " but not listed among the known modules. This will not work under OSGi or JBoss vfs.");
            }
        }
        Enumeration<URL> kieConfEnum = cl.getResources(LEGACY_CONF_FILE);
        while (kieConfEnum.hasMoreElements()) {
            kieConfsUrls.add(kieConfEnum.nextElement());
        }
        if (log.isDebugEnabled()) {
            log.debug("Discovered kie.conf files: " + kieConfsUrls);
        }
        return kieConfsUrls;
    }

    public static Stream<URL> getKieConfsFromKnownModules(ClassLoader cl) {
        return Stream.of(KIE_MODULES).map(module -> cl.getResource("META-INF/kie/" + module + (module.length() > 0 ? "/" : "") + CONF_FILE_NAME)).filter(Objects::nonNull);
    }

    private static void collectKieConfsInJar(List<URL> kieConfsUrls, URL metaInf, JarURLConnection con) throws IOException {
        JarURLConnection jarCon = con;
        JarFile jarFile = jarCon.getJarFile();
        Enumeration<JarEntry> entries = jarFile.entries();
        while (entries.hasMoreElements()) {
            JarEntry entry = entries.nextElement();
            if (!entry.getName().endsWith(CONF_FILE_NAME)) continue;
            String metaInfString = metaInf.toString();
            int confFileFolderLength = metaInfString.endsWith("/") ? CONF_FILE_FOLDER.length() + 1 : CONF_FILE_FOLDER.length();
            kieConfsUrls.add(new URL(metaInfString.substring(0, metaInfString.length() - confFileFolderLength) + entry.getName()));
        }
    }

    private static void collectKieConfsInFile(List<URL> kieConfsUrls, File file) throws IOException {
        if (file.isDirectory()) {
            for (File child : file.listFiles()) {
                ServiceDiscoveryImpl.collectKieConfsInFile(kieConfsUrls, child);
            }
        } else if (file.toString().endsWith(CONF_FILE_NAME)) {
            kieConfsUrls.add(file.toURI().toURL());
        }
    }

    private static String getModuleName(URL url) {
        String s = url.toString();
        int moduleStart = s.indexOf(CONF_FILE_FOLDER) + CONF_FILE_FOLDER.length() + 1;
        return s.substring(moduleStart, s.length() - (CONF_FILE_NAME.length() + 1));
    }

    private static class PriorityMap<K, V> {
        private final Map<K, TreeMap<Integer, V>> priorityMap = new HashMap<K, TreeMap<Integer, V>>();

        private PriorityMap() {
        }

        public void reset() {
            this.priorityMap.clear();
        }

        public void put(int priority, K key, V value) {
            TreeMap<Integer, V> treeMap = this.priorityMap.get(key);
            if (treeMap == null) {
                treeMap = new TreeMap();
                this.priorityMap.put(key, treeMap);
            } else if (treeMap.get(priority) != null) {
                throw new RuntimeException("There already exists an implementation for service " + key + " with same priority " + priority);
            }
            treeMap.put(priority, value);
        }

        public Iterable<? extends Map.Entry<K, List<V>>> entrySet() {
            HashMap map = new HashMap();
            for (Map.Entry<K, TreeMap<Integer, V>> entry : this.priorityMap.entrySet()) {
                ArrayList<V> list = new ArrayList<V>();
                for (V value : entry.getValue().values()) {
                    list.add(0, value);
                }
                map.put(entry.getKey(), list);
            }
            return map.entrySet();
        }
    }

    private static class KieConfs {
        private final ClassLoader classLoader;
        private final Collection<URL> resources;

        private KieConfs(ClassLoader classLoader, Collection<URL> confResources) {
            this.classLoader = classLoader;
            this.resources = confResources;
        }
    }

    private static class LazyHolder {
        static final ServiceDiscoveryImpl INSTANCE = new ServiceDiscoveryImpl();

        private LazyHolder() {
        }
    }
}

