/*
 * Copyright 2021 Red Hat, Inc. and/or its affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.acme.vehiclerouting.domain.geo;

import static java.lang.Math.ceil;
import static java.lang.Math.sqrt;

import org.acme.vehiclerouting.domain.Location;

public class EuclideanDistanceCalculator implements DistanceCalculator {

    // Approximate Metric Equivalents for Degrees. At the equator for longitude and for latitude anywhere,
    // the following approximations are valid: 1° = 111 km (or 60 nautical miles) 0.1° = 11.1 km.
    public static final long METERS_PER_DEGREE = 111_000;

    @Override
    public long calculateDistance(Location from, Location to) {
        if (from.equals(to)) {
            return 0L;
        }
        double latitudeDiff = to.getLatitude() - from.getLatitude();
        double longitudeDiff = to.getLongitude() - from.getLongitude();
        return (long) ceil(sqrt(latitudeDiff * latitudeDiff + longitudeDiff * longitudeDiff) * METERS_PER_DEGREE);
    }
}
