/*
 * JBoss, Home of Professional Open Source.
 * See the COPYRIGHT.txt file distributed with this work for information
 * regarding copyright ownership.  Some portions may be licensed
 * to Red Hat, Inc. under one or more contributor license agreements.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301 USA.
 */
package org.komodo.rest;

import java.util.Base64;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;

import org.komodo.metadata.DefaultMetadataInstance;
import org.komodo.metadata.TeiidConnectionProvider;
import org.komodo.spi.KException;
import org.komodo.spi.runtime.TeiidDataSource;
import org.teiid.adminapi.AdminException;

import com.fasterxml.jackson.databind.ObjectMapper;

import io.kubernetes.client.ModelServiceCatalogClient;
import io.kubernetes.client.Secret;
import io.kubernetes.client.ServiceInstance;
import io.kubernetes.client.ServiceInstanceList;

public class TeiidSwarmMetadataInstance extends DefaultMetadataInstance {

	private static final String OSURL = "https://openshift.default.svc";
	private static final String SC_VERSION = "v1beta1";
	private ModelServiceCatalogClient scClient;
	
    public TeiidSwarmMetadataInstance(TeiidConnectionProvider connectionProvider) {
		super(connectionProvider);
		this.scClient = new ModelServiceCatalogClient(OSURL, SC_VERSION);
	}
    
    @Override
    public Collection<TeiidDataSource> getDataSources() throws KException {
        checkStarted();
        try {
        	Collection<String> dsNames = admin().getDataSourceNames();
        	String token = AuthHandlingFilter.threadOAuthCredentials.get().getToken();
        	System.out.println("Access token = "+token);
        	if (token != null) {
        		this.scClient.setAuthHeader(token);
        		
        		ServiceInstanceList serviceList = this.scClient.getServiceInstances(ApplicationProperties.getNamespace());
        		List<ServiceInstance> services = serviceList.getItems();
        		if( services != null && !services.isEmpty()) {
        			for (ServiceInstance svc:services) {
        				String name = svc.getMetadata().getName();
        				System.out.println("Service Name = "+name);
        				if (svc.getStatus().isReady()) {
        					// data source is there.
        					String secretName = svc.getSpec().getParametersFrom().getSecretKeyRef().getName();
        					String key = svc.getSpec().getParametersFrom().getSecretKeyRef().getKey();
							if (!dsNames.contains(name)) {
								Secret secret = this.scClient.getSecret(ApplicationProperties.getNamespace(),
										secretName);
								if (secret != null) {
									String json = secret.getData().get(key);
									Map<String, String> map = new ObjectMapper().readerFor(Map.class)
											.readValue(Base64.getDecoder().decode(json));
									createDataSource(name, map);
								}
							}
        				}
        			}
        		}
        	}
            return super.getDataSources();
        } catch (Exception ex) {
            throw handleError(ex);
        }
    }

	private void createDataSource(String name, Map<String, String> map) throws AdminException {
		if (map.get("POSTGRESQL_DATABASE") != null) {
			System.out.println("Creating the Datasource = "+name);
			/*
			 {
			   "DATABASE_SERVICE_NAME":"postgresql",
			   "MEMORY_LIMIT":"512Mi",
			   "NAMESPACE":"openshift",
			   "POSTGRESQL_DATABASE":"sampledb",
			   "POSTGRESQL_PASSWORD":"pass",
			   "POSTGRESQL_USER":"user",
			   "POSTGRESQL_VERSION":"9.5",
			   "VOLUME_CAPACITY":"1Gi"
			}
			*/
			String driverName = null;
			Set<String> templateNames = admin().getDataSourceTemplateNames();
			for (String template : templateNames) {
				if (template.startsWith("postgresql")) {
					driverName = template;
				}
			}
			Properties props = new Properties();
			props.setProperty("connection-url", "jdbc:postgresql://" + map.get("DATABASE_SERVICE_NAME") + ":5432/"
					+ map.get("POSTGRESQL_DATABASE"));
			props.setProperty("user-name", map.get("POSTGRESQL_USER"));
			props.setProperty("password", map.get("POSTGRESQL_PASSWORD"));
			admin().createDataSource(name, driverName, props);
		}
	}
    
}
