/*
 * Decompiled with CFR 0.152.
 */
package org.apache.accumulo.core.file.blockfile.cache;

import java.lang.ref.WeakReference;
import java.util.Objects;
import java.util.PriorityQueue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.locks.ReentrantLock;
import org.apache.accumulo.core.file.blockfile.cache.BlockCache;
import org.apache.accumulo.core.file.blockfile.cache.CacheEntry;
import org.apache.accumulo.core.file.blockfile.cache.CachedBlock;
import org.apache.accumulo.core.file.blockfile.cache.CachedBlockQueue;
import org.apache.accumulo.core.file.blockfile.cache.ClassSize;
import org.apache.accumulo.core.file.blockfile.cache.HeapSize;
import org.apache.accumulo.core.util.NamingThreadFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LruBlockCache
implements BlockCache,
HeapSize {
    private static final Logger log = LoggerFactory.getLogger(LruBlockCache.class);
    static final float DEFAULT_LOAD_FACTOR = 0.75f;
    static final int DEFAULT_CONCURRENCY_LEVEL = 16;
    static final float DEFAULT_MIN_FACTOR = 0.75f;
    static final float DEFAULT_ACCEPTABLE_FACTOR = 0.85f;
    static final float DEFAULT_SINGLE_FACTOR = 0.25f;
    static final float DEFAULT_MULTI_FACTOR = 0.5f;
    static final float DEFAULT_MEMORY_FACTOR = 0.25f;
    static final int statThreadPeriod = 60;
    private final ConcurrentHashMap<String, CachedBlock> map;
    private final ReentrantLock evictionLock = new ReentrantLock(true);
    private volatile boolean evictionInProgress = false;
    private final EvictionThread evictionThread;
    private final ScheduledExecutorService scheduleThreadPool = Executors.newScheduledThreadPool(1, new NamingThreadFactory("LRUBlockCacheStats"));
    private final AtomicLong size;
    private final AtomicLong elements;
    private final AtomicLong count;
    private final CacheStats stats;
    private long maxSize;
    private long blockSize;
    private float acceptableFactor;
    private float minFactor;
    private float singleFactor;
    private float multiFactor;
    private float memoryFactor;
    private long overhead;
    public static final long CACHE_FIXED_OVERHEAD = ClassSize.align(24 + 8 * ClassSize.REFERENCE + 20 + 1 + ClassSize.OBJECT);

    public LruBlockCache(long maxSize, long blockSize) {
        this(maxSize, blockSize, true);
    }

    public LruBlockCache(long maxSize, long blockSize, boolean evictionThread) {
        this(maxSize, blockSize, evictionThread, (int)Math.ceil(1.2 * (double)maxSize / (double)blockSize), 0.75f, 16, 0.75f, 0.85f, 0.25f, 0.5f, 0.25f);
    }

    public LruBlockCache(long maxSize, long blockSize, boolean evictionThread, int mapInitialSize, float mapLoadFactor, int mapConcurrencyLevel, float minFactor, float acceptableFactor, float singleFactor, float multiFactor, float memoryFactor) {
        if (singleFactor + multiFactor + memoryFactor != 1.0f) {
            throw new IllegalArgumentException("Single, multi, and memory factors  should total 1.0");
        }
        if (minFactor >= acceptableFactor) {
            throw new IllegalArgumentException("minFactor must be smaller than acceptableFactor");
        }
        if (minFactor >= 1.0f || acceptableFactor >= 1.0f) {
            throw new IllegalArgumentException("all factors must be < 1");
        }
        this.maxSize = maxSize;
        this.blockSize = blockSize;
        this.map = new ConcurrentHashMap(mapInitialSize, mapLoadFactor, mapConcurrencyLevel);
        this.minFactor = minFactor;
        this.acceptableFactor = acceptableFactor;
        this.singleFactor = singleFactor;
        this.multiFactor = multiFactor;
        this.memoryFactor = memoryFactor;
        this.stats = new CacheStats();
        this.count = new AtomicLong(0L);
        this.elements = new AtomicLong(0L);
        this.overhead = LruBlockCache.calculateOverhead(maxSize, blockSize, mapConcurrencyLevel);
        this.size = new AtomicLong(this.overhead);
        if (evictionThread) {
            this.evictionThread = new EvictionThread(this);
            this.evictionThread.start();
            while (!this.evictionThread.running()) {
                try {
                    Thread.sleep(10L);
                }
                catch (InterruptedException ex) {
                    throw new RuntimeException(ex);
                }
            }
        } else {
            this.evictionThread = null;
        }
        this.scheduleThreadPool.scheduleAtFixedRate(new StatisticsThread(this), 60L, 60L, TimeUnit.SECONDS);
    }

    public void setMaxSize(long maxSize) {
        this.maxSize = maxSize;
        if (this.size.get() > this.acceptableSize() && !this.evictionInProgress) {
            this.runEviction();
        }
    }

    @Override
    public CacheEntry cacheBlock(String blockName, byte[] buf, boolean inMemory) {
        CachedBlock cb = this.map.get(blockName);
        if (cb != null) {
            this.stats.duplicateReads();
            cb.access(this.count.incrementAndGet());
        } else {
            cb = new CachedBlock(blockName, buf, this.count.incrementAndGet(), inMemory);
            CachedBlock currCb = this.map.putIfAbsent(blockName, cb);
            if (currCb != null) {
                this.stats.duplicateReads();
                cb = currCb;
                cb.access(this.count.incrementAndGet());
            } else {
                long newSize = this.size.addAndGet(cb.heapSize());
                this.elements.incrementAndGet();
                if (newSize > this.acceptableSize() && !this.evictionInProgress) {
                    this.runEviction();
                }
            }
        }
        return cb;
    }

    @Override
    public CacheEntry cacheBlock(String blockName, byte[] buf) {
        return this.cacheBlock(blockName, buf, false);
    }

    @Override
    public CachedBlock getBlock(String blockName) {
        CachedBlock cb = this.map.get(blockName);
        if (cb == null) {
            this.stats.miss();
            return null;
        }
        this.stats.hit();
        cb.access(this.count.incrementAndGet());
        return cb;
    }

    protected long evictBlock(CachedBlock block) {
        this.map.remove(block.getName());
        this.size.addAndGet(-1L * block.heapSize());
        this.elements.decrementAndGet();
        this.stats.evicted();
        return block.heapSize();
    }

    private void runEviction() {
        if (this.evictionThread == null) {
            this.evict();
        } else {
            this.evictionThread.evict();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void evict() {
        if (!this.evictionLock.tryLock()) {
            return;
        }
        try {
            BlockBucket bucket;
            this.evictionInProgress = true;
            long bytesToFree = this.size.get() - this.minSize();
            log.trace("Block cache LRU eviction started.  Attempting to free {} bytes", (Object)bytesToFree);
            if (bytesToFree <= 0L) {
                return;
            }
            BlockBucket bucketSingle = new BlockBucket(bytesToFree, this.blockSize, this.singleSize());
            BlockBucket bucketMulti = new BlockBucket(bytesToFree, this.blockSize, this.multiSize());
            BlockBucket bucketMemory = new BlockBucket(bytesToFree, this.blockSize, this.memorySize());
            for (CachedBlock cachedBlock : this.map.values()) {
                switch (cachedBlock.getPriority()) {
                    case SINGLE: {
                        bucketSingle.add(cachedBlock);
                        break;
                    }
                    case MULTI: {
                        bucketMulti.add(cachedBlock);
                        break;
                    }
                    case MEMORY: {
                        bucketMemory.add(cachedBlock);
                    }
                }
            }
            PriorityQueue<BlockBucket> bucketQueue = new PriorityQueue<BlockBucket>(3);
            bucketQueue.add(bucketSingle);
            bucketQueue.add(bucketMulti);
            bucketQueue.add(bucketMemory);
            int remainingBuckets = 3;
            long bytesFreed = 0L;
            while ((bucket = (BlockBucket)bucketQueue.poll()) != null) {
                long overflow = bucket.overflow();
                if (overflow > 0L) {
                    long bucketBytesToFree = Math.min(overflow, (long)Math.ceil((double)(bytesToFree - bytesFreed) / (double)remainingBuckets));
                    bytesFreed += bucket.free(bucketBytesToFree);
                }
                --remainingBuckets;
            }
            float singleMB = (float)bucketSingle.totalSize() / 1048576.0f;
            float multiMB = (float)bucketMulti.totalSize() / 1048576.0f;
            float memoryMB = (float)bucketMemory.totalSize() / 1048576.0f;
            log.trace("Block cache LRU eviction completed. Freed {} bytes. Priority Sizes: Single={}MB ({}), Multi={}MB ({}), Memory={}MB ({})", new Object[]{bytesFreed, Float.valueOf(singleMB), bucketSingle.totalSize(), Float.valueOf(multiMB), bucketMulti.totalSize(), Float.valueOf(memoryMB), bucketMemory.totalSize()});
        }
        finally {
            this.stats.evict();
            this.evictionInProgress = false;
            this.evictionLock.unlock();
        }
    }

    @Override
    public long getMaxSize() {
        return this.maxSize;
    }

    public long getCurrentSize() {
        return this.size.get();
    }

    public long getFreeSize() {
        return this.getMaxSize() - this.getCurrentSize();
    }

    public long size() {
        return this.elements.get();
    }

    public long getEvictionCount() {
        return this.stats.getEvictionCount();
    }

    public long getEvictedCount() {
        return this.stats.getEvictedCount();
    }

    public void logStats() {
        long totalSize = this.heapSize();
        long freeSize = this.maxSize - totalSize;
        float sizeMB = (float)totalSize / 1048576.0f;
        float freeMB = (float)freeSize / 1048576.0f;
        float maxMB = (float)this.maxSize / 1048576.0f;
        log.debug("Cache Stats: Sizes: Total={}MB ({}), Free={}MB ({}), Max={}MB ({}), Counts: Blocks={}, Access={}, Hit={}, Miss={}, Evictions={}, Evicted={},Ratios: Hit Ratio={}%, Miss Ratio={}%, Evicted/Run={}, Duplicate Reads={}", new Object[]{Float.valueOf(sizeMB), totalSize, Float.valueOf(freeMB), freeSize, Float.valueOf(maxMB), this.maxSize, this.size(), this.stats.getRequestCount(), this.stats.getHitCount(), this.stats.getMissCount(), this.stats.getEvictionCount(), this.stats.getEvictedCount(), this.stats.getHitRatio() * 100.0, this.stats.getMissRatio() * 100.0, this.stats.evictedPerEviction(), this.stats.getDuplicateReads()});
    }

    public CacheStats getStats() {
        return this.stats;
    }

    @Override
    public long heapSize() {
        return this.getCurrentSize();
    }

    public static long calculateOverhead(long maxSize, long blockSize, int concurrency) {
        return CACHE_FIXED_OVERHEAD + (long)ClassSize.CONCURRENT_HASHMAP + (long)((int)Math.ceil((double)maxSize * 1.2 / (double)blockSize) * ClassSize.CONCURRENT_HASHMAP_ENTRY) + (long)(concurrency * ClassSize.CONCURRENT_HASHMAP_SEGMENT);
    }

    private long acceptableSize() {
        return (long)Math.floor((float)this.maxSize * this.acceptableFactor);
    }

    private long minSize() {
        return (long)Math.floor((float)this.maxSize * this.minFactor);
    }

    private long singleSize() {
        return (long)Math.floor((float)this.maxSize * this.singleFactor * this.minFactor);
    }

    private long multiSize() {
        return (long)Math.floor((float)this.maxSize * this.multiFactor * this.minFactor);
    }

    private long memorySize() {
        return (long)Math.floor((float)this.maxSize * this.memoryFactor * this.minFactor);
    }

    public void shutdown() {
        this.scheduleThreadPool.shutdown();
    }

    public static class CacheStats {
        private final AtomicLong accessCount = new AtomicLong(0L);
        private final AtomicLong hitCount = new AtomicLong(0L);
        private final AtomicLong missCount = new AtomicLong(0L);
        private final AtomicLong evictionCount = new AtomicLong(0L);
        private final AtomicLong evictedCount = new AtomicLong(0L);
        private final AtomicLong duplicateReads = new AtomicLong(0L);

        public void miss() {
            this.missCount.incrementAndGet();
            this.accessCount.incrementAndGet();
        }

        public void hit() {
            this.hitCount.incrementAndGet();
            this.accessCount.incrementAndGet();
        }

        public void evict() {
            this.evictionCount.incrementAndGet();
        }

        public void duplicateReads() {
            this.duplicateReads.incrementAndGet();
        }

        public void evicted() {
            this.evictedCount.incrementAndGet();
        }

        public long getRequestCount() {
            return this.accessCount.get();
        }

        public long getMissCount() {
            return this.missCount.get();
        }

        public long getHitCount() {
            return this.hitCount.get();
        }

        public long getEvictionCount() {
            return this.evictionCount.get();
        }

        public long getDuplicateReads() {
            return this.duplicateReads.get();
        }

        public long getEvictedCount() {
            return this.evictedCount.get();
        }

        public double getHitRatio() {
            return (float)this.getHitCount() / (float)this.getRequestCount();
        }

        public double getMissRatio() {
            return (float)this.getMissCount() / (float)this.getRequestCount();
        }

        public double evictedPerEviction() {
            return (float)this.getEvictedCount() / (float)this.getEvictionCount();
        }
    }

    private static class StatisticsThread
    extends Thread {
        LruBlockCache lru;

        public StatisticsThread(LruBlockCache lru) {
            super("LruBlockCache.StatisticsThread");
            this.setDaemon(true);
            this.lru = lru;
        }

        @Override
        public void run() {
            this.lru.logStats();
        }
    }

    private static class EvictionThread
    extends Thread {
        private WeakReference<LruBlockCache> cache;
        private boolean running = false;

        public EvictionThread(LruBlockCache cache) {
            super("LruBlockCache.EvictionThread");
            this.setDaemon(true);
            this.cache = new WeakReference<LruBlockCache>(cache);
        }

        public synchronized boolean running() {
            return this.running;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void run() {
            while (true) {
                EvictionThread evictionThread = this;
                synchronized (evictionThread) {
                    this.running = true;
                    try {
                        this.wait();
                    }
                    catch (InterruptedException interruptedException) {
                        // empty catch block
                    }
                }
                LruBlockCache cache = (LruBlockCache)this.cache.get();
                if (cache == null) break;
                cache.evict();
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public void evict() {
            EvictionThread evictionThread = this;
            synchronized (evictionThread) {
                this.notify();
            }
        }
    }

    private class BlockBucket
    implements Comparable<BlockBucket> {
        private CachedBlockQueue queue;
        private long totalSize = 0L;
        private long bucketSize;

        public BlockBucket(long bytesToFree, long blockSize, long bucketSize) {
            this.bucketSize = bucketSize;
            this.queue = new CachedBlockQueue(bytesToFree, blockSize);
            this.totalSize = 0L;
        }

        public void add(CachedBlock block) {
            this.totalSize += block.heapSize();
            this.queue.add(block);
        }

        public long free(long toFree) {
            CachedBlock[] blocks = this.queue.get();
            long freedBytes = 0L;
            for (int i = 0; i < blocks.length; ++i) {
                if ((freedBytes += LruBlockCache.this.evictBlock(blocks[i])) < toFree) continue;
                return freedBytes;
            }
            return freedBytes;
        }

        public long overflow() {
            return this.totalSize - this.bucketSize;
        }

        public long totalSize() {
            return this.totalSize;
        }

        @Override
        public int compareTo(BlockBucket that) {
            if (this.overflow() == that.overflow()) {
                return 0;
            }
            return this.overflow() > that.overflow() ? 1 : -1;
        }

        public int hashCode() {
            return Objects.hashCode(this.overflow());
        }

        public boolean equals(Object that) {
            if (that instanceof BlockBucket) {
                return this.compareTo((BlockBucket)that) == 0;
            }
            return false;
        }
    }
}

