/*
	Copyright 2009 Anatol Gregory Mayen
	
	Licensed under the Apache License, Version 2.0 (the "License");
	you may not use this file except in compliance with the License. 
	You may obtain a copy of the License at 
	
	http://www.apache.org/licenses/LICENSE-2.0 
	
	Unless required by applicable law or agreed to in writing, software 
	distributed under the License is distributed on an "AS IS" BASIS, 
	WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
	See the License for the specific language governing permissions and 
	limitations under the License. 
*/
package eu.maydu.gwt.validation.client;

import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.google.gwt.core.client.GWT;
import com.google.gwt.user.client.Window;

import eu.maydu.gwt.validation.client.ValidationResult.ValidationError;
import eu.maydu.gwt.validation.client.exceptions.IncorrectSerializationFormatException;
import eu.maydu.gwt.validation.client.i18n.StandardValidationMessages;
import eu.maydu.gwt.validation.client.i18n.ValidationMessages;

/**
 * The <code>ValidationProcessor</code> is the core of the client side validation functionality.
 * It is responsible for invoking the validations and the actions for the failed validators.
 * 
 * It is also used to process the server side validations, if they are passed to the
 * <code>processServerErrors</code> method.
 * 
 * A default implementation is provided via the {@link DefaultValidationProcessor}.
 * 
 * @author Anatol Mayen
 *
 */
public interface ValidationProcessor {

	/**
	 * Errors coming from the server side prefixed with this string are assumed coming from hibernate
	 */
	public static final String HIBERNATE_VALIDATION_ERROR_PREFIX = "xxxHibernateErrorxxx";
	
	/**
	 * Errors coming from the server side prefixed with this string are assumed to be standard error messages
	 */
	public static final String SERVER_SIDE_STANDARD_PREFIX = "xxxStandardErrorxxx";


	/**
	 * The prefix of the key given to the <code>ValidationMessages.getPropertyName</code> method
	 */
	public static String PROPERTY_NAME_LOCALIZATION_PREFIX = "property_";
	

	/**
	 * Add new validators that the <code>ValidationProcessor</code> instance will
	 * validate.
	 * 
	 * The <code>name</code> parameter is used to connect the client side with the server side validations.
	 * If you want to validate a field on the server side you need to specify the same <code>name</code>
	 * for the input field here on this method and on the validation on the server side. E.g.
	 * on the server side you might have something like this:
	 * <pre>
	 * 		new ServerValidation().inRange(car.getPs(), 0, 350, "ps");
	 * </pre>
	 * If you want the error to be shown on the correct input field on the client side
	 * the <code>name</code> parameter must be the same as defined on the server side
	 * ("ps" in this case).
	 * 
	 * 
	 * 
	 * @param name The name that is used to connect client and server side validations
	 * @param validator The validators that will be used for validations
	 */
	public ValidationProcessor addValidators(String name, Validator... validator);
	
	/**
	 * Adds an global action to this <code>ValidatorProcessor</code> instance.
	 * This is primarily useful for actions that require all the validation errors
	 * of a validation run. The actions added as global actions will only be 
	 * invoked <i>after</i> all validations and their associated actions are invoked
	 * and only if there were validation errors during the validation run. 
	 * 
	 * @param action The global action
	 * @return A reference to this <code>ValidationProcessor</code>'s instance for chaining.
	 */
	public ValidationProcessor addGlobalAction(ValidationAction action);
	
	
	/**
	 * Invoke the client side only validation.
	 * 
	 * 
	 * @param names The names of the validations that should be validated, can be <code>null</code> or not given to validate everything.
	 * @return <code>true</code> if all client side validations were successful, <code>false</code> otherwise
	 */
	public boolean validate(String... names);

	/**
	 * Processes the validation errors that came from the server side
	 * and displays them on the client side.
	 * 
	 * @see eu.maydu.gwt.validation.server.ServerValidation
	 *  
	 * @param validationException
	 * @return <i>true</i> if no errors were included in the given <code>ValidationException</code>
	 */
	public boolean processServerErrors(ValidationException validationException);
	
	/**
	 * If you got a ValidationException in serialized form (if you use restlet for example)
	 * you can pass it to this method and the ValidationProcessor will deserialize and process
	 * it, just like the processServerErrors method which is used for processing the errors
	 * you got from RPC directly as a ValidationException object.
	 * 
	 * The serialized ValidationException mechanism should be used when you can not (or don't want
	 * to) use the RPC mechanism that directly passes ValidationException's around.
	 * 
	 * @param serializedValidationException
	 * @return
	 */
	public boolean processSerializedValidationErrors(String serializedValidationException);
	
	/**
	 * 
	 * This method checks if an arbitrary String is a serialized ValidationException.
	 * This method will just silently quit and return a value of
	 * <i>true</i> if the given String was not a serialized <code>ValidationException</code>.
	 * If it was a serialized <code>ValidationException</code> it will process it.
	 * 
	 * This is useful in cases where you don't know if the response denotes a <code>ValidationException</code>
	 * or not. It will effectively just do its job when the given string is a <code>ValidationException</code>
	 * and will say that everything is ok if the given string was not a <code>ValidationException</code>.
	 * 
	 * 
	 * @param serializedValidationException The string that MIGHT be a serialized <code>ValidationException</code>
	 * @return <i>true</i> If there were no errors (String was no serialized <code>ValidationException</code> or it was but did not contain any errors.
	 */
	public boolean processSerializedValidationErrorsSilently(String serializedValidationException);
	
	public ValidationException deserializeValidationException(
			String serializedValidationException);



	/**
	 * Resets all the actions specified that were invoked by a possibly failed previous validation run.
	 * 
	 * @param names Specify the validations that should be reset, can be <code>null</code> to reset everything.
	 */
	public void reset(String... names);
	

	/**
	 * 
	 * @return The count of all added validators.
	 */
	public int getValidatorCount();
	
	
	/**
	 * 
	 * @return The count fo all added global actions.
	 */
	public int getGlobalActionCount();
	
	/**
	 * Removes all validators.
	 */
	public void removeAllValidators();
	
	/**
	 * Removes all global actions.
	 */
	public void removeAllGlobalActions();
	
	/**
	 * Resets the <code>ValidationProcessor</code>.
	 * 
	 * Invokes <code>removeAllValidators</code>
	 * and <code>removeAllGlobalActions</code>
	 */
	public void removeValidatorsAndGlobalActions();
	
	/**
	 * Removes all the validators associated with the given names
	 * 
	 * @param validatorNames The names of the added validators to delete
	 */
	public void removeValidators(String... validatorNames);
	
	/**
	 * Removes all the given actions from the global actions list
	 *  
	 * @param instancesToRemove The instances to remove
	 */
	public void removeGlobalActions(ValidationAction... instancesToRemove);
	
	/**
	 * If you want to add a prefix to the property names
	 * that will be handed over to the <code>ValidationMessages.getPropertyName()</code>
	 * method for localization set this to <code>true</code> default is <code>false</code>
	 * 
	 * The prefix will be: <code>ValidationProcessor.PROPERTY_NAME_LOCALIZATION_PREFIX</code>
	 * 
	 * @param addPrefix if <code>true</code> a prefix will be added to the property name on property localization
	 */
	public void setPropertyLocalizationPrefixPrepended(boolean addPrefix);
	
	/**
	 * Is property name prefixing enabled?
	 * 
	 * @return <i>true</i> if property name prefixing is enabled, <i>false</i> otherwise.
	 */
	public boolean isPropertyLocalizationPrefixPrepended();
}
