/*
	Copyright 2009 Anatol Gregory Mayen
	
	Licensed under the Apache License, Version 2.0 (the "License");
	you may not use this file except in compliance with the License. 
	You may obtain a copy of the License at 
	
	http://www.apache.org/licenses/LICENSE-2.0 
	
	Unless required by applicable law or agreed to in writing, software 
	distributed under the License is distributed on an "AS IS" BASIS, 
	WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
	See the License for the specific language governing permissions and 
	limitations under the License. 
*/
package eu.maydu.gwt.validation.client.description;

import com.google.gwt.i18n.client.LocaleInfo;
import com.google.gwt.user.client.ui.FocusListener;
import com.google.gwt.user.client.ui.FocusWidget;
import com.google.gwt.user.client.ui.HTML;
import com.google.gwt.user.client.ui.PopupPanel;
import com.google.gwt.user.client.ui.Widget;
import com.google.gwt.user.client.ui.PopupPanel.PositionCallback;

import eu.maydu.gwt.validation.client.i18n.ValidationMessages;

/**
 * Creates an popup that is shown whenever the associated widget
 * gets the focus. The popup will contain arbitrary HTML interpreted
 * text that can easily be i18n.
 * 
 * The popup is shown directly under the associated widget. It will
 * autohide if the user clicks somewhere or moves the focus to another
 * component.
 * 
 * @author Anatol Mayen
 *
 */
public class PopupDescription implements Description<FocusWidget> {

	private ValidationMessages messages;
	private LocaleInfo localeInfo = LocaleInfo.getCurrentLocale();
	
	/**
	 * Creates a new PopupDescription and configures it with the specified 
	 * <code>ValidationMessages</code>. Be sure to overwrite the 
	 * <code>getDescriptionMessage</code> method to i18n the description
	 * text. 
	 * 
	 * 
	 * @param messages The validation messages instance to use 
	 */
	public PopupDescription(ValidationMessages messages) {
		this.messages = messages;
	}
	

	/**
	 * Adds a popup description to a widget.
	 * 
	 * @param key This key is used to get the i18n text that should be displayed. This key will be passed as argument to the <code>ValidationMessages.getDescriptionMessage</code> method.
	 * @param widget The widget that should show a popup description when it gets the focus.
	 */
	public void addDescription(String key, final FocusWidget widget) {
		
		final PopupPanel p = new PopupPanel(true);
		
		String content = messages.getDescriptionMessage(key);
		
		if(localeInfo.isRTL())
			content = "<div align=\"right\">" + content + "</div>";
		
		HTML html = new HTML(content, false);
		p.setWidget(html);
		
		widget.addFocusListener(new FocusListener() {
			public void onFocus(Widget sender) {
				p.setPopupPositionAndShow(new PositionCallback() {

					public void setPosition(int offsetWidth, int offsetHeight) {
						
						int left, top, height;
						
						if(!localeInfo.isRTL()) {
							left = widget.getAbsoluteLeft();
						}else {
							left = widget.getAbsoluteLeft()+widget.getOffsetWidth();
							left -= p.getOffsetWidth();
						}
						top = widget.getAbsoluteTop();
						height = widget.getOffsetHeight();

						p.setPopupPosition(left, top+height+3);
					}
				});
			}

			public void onLostFocus(Widget sender) {
				p.hide();
			}
		});
	}
}
