/*
	Copyright 2009 Anatol Gregory Mayen
	
	Licensed under the Apache License, Version 2.0 (the "License");
	you may not use this file except in compliance with the License. 
	You may obtain a copy of the License at 
	
	http://www.apache.org/licenses/LICENSE-2.0 
	
	Unless required by applicable law or agreed to in writing, software 
	distributed under the License is distributed on an "AS IS" BASIS, 
	WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
	See the License for the specific language governing permissions and 
	limitations under the License. 
*/
package eu.maydu.gwt.validation.client.validators.standard;

import com.google.gwt.user.client.ui.SuggestBox;
import com.google.gwt.user.client.ui.TextBoxBase;

import eu.maydu.gwt.validation.client.ValidationAction;
import eu.maydu.gwt.validation.client.ValidationResult;
import eu.maydu.gwt.validation.client.Validator;
import eu.maydu.gwt.validation.client.i18n.ValidationMessages;
import eu.maydu.gwt.validation.client.i18n.utils.MessageInterpolator;
import eu.maydu.gwt.validation.client.validators.ValidatorConfigurationSource;

/**
 * 
 * Validates that the input corresponds to the given
 * regular expression.
 * 
 * This validator needs to be given a custom message key
 * because this validator is so general that a single message
 * would not be enough. 
 * 
 * @author Anatol Mayen
 *
 */
public class RegularExpressionValidator extends Validator<RegularExpressionValidator> {

	
	private TextBoxBase textBox = null;
	private SuggestBox suggestBox = null;
	private ValidatorConfigurationSource<String> configSource = null;
	private String regexPattern = null;
	
	
	/**
	 * 
	 * Creates a new <code>RegularExpressionValidator</code> that validates
	 * the given input against the given regular expression. There is no 
	 * default validation error message associated with this <code>Validator</code>.
	 * 
	 * The localized error message may choose to include a {0} place holder. If
	 * it chooses to, it will be replaced by the given input string on validation
	 * in the associated <code>TextBoxBase</code> field.
	 * 
	 * 
	 * @param text The input field to validate
	 * @param regexPattern The pattern after which to validate
	 * @param customMsgKey The key of a custom message
	 */
	public RegularExpressionValidator(TextBoxBase text, String regexPattern, String customMsgKey ) {
		this.textBox = text;
		this.regexPattern = regexPattern;
		this.setCustomMsgKey(customMsgKey);
	}
	
	public RegularExpressionValidator(SuggestBox suggest, String regexPattern, String customMsgKey) {
		this.suggestBox = suggest;
		this.regexPattern = regexPattern;
		this.setCustomMsgKey(customMsgKey);
	}

	public RegularExpressionValidator(TextBoxBase text, ValidatorConfigurationSource<String> configSource, String customMsgKey) {
		this.textBox = text;
		this.configSource = configSource;
		this.setCustomMsgKey(customMsgKey);
	}
	
	public RegularExpressionValidator(SuggestBox suggest, ValidatorConfigurationSource<String> configSource, String customMsgKey) {
		this.suggestBox = suggest;
		this.configSource = configSource;
		this.setCustomMsgKey(customMsgKey);
	}

	
	
	
	@Override
	public void invokeActions(ValidationResult result) {
		if(textBox != null) {
			for(ValidationAction<TextBoxBase> va : this.getFailureActions())
				va.invoke(result, textBox);
		}else {
			for(ValidationAction<SuggestBox> va : this.getFailureActions())
				va.invoke(result, suggestBox);
		}
			
		
	}

	@Override
	public <V extends ValidationMessages> ValidationResult validate(V messages) {
		
		String text;
		if(suggestBox != null)
			text = suggestBox.getText();
		else text = textBox.getText();
	
		if(text.equals("") && !isRequired())
			return null;
		String regex;
		if(configSource == null)
			regex = regexPattern;
		else
			regex = configSource.getConfigurationValue();
		try {
		if(!text.matches(regex))
			return new ValidationResult(getErrorMessage(messages, messages.getStandardMessages().regexNotFulfilled(),text, regex));
		}catch(IllegalArgumentException ex) {
			return new ValidationResult(messages.getStandardMessages().notARegEx());
		}
		
		return null;
	}

}