/*
 * Decompiled with CFR 0.152.
 */
package io.agroal.pool;

import io.agroal.api.configuration.AgroalConnectionPoolConfiguration;
import io.agroal.api.transaction.TransactionIntegration;
import io.agroal.pool.ConnectionFactory;
import io.agroal.pool.ConnectionHandler;
import io.agroal.pool.DataSource;
import io.agroal.pool.ListenerHelper;
import io.agroal.pool.util.AgroalSynchronizer;
import io.agroal.pool.util.PriorityScheduledExecutor;
import io.agroal.pool.util.StampedCopyOnWriteArrayList;
import io.agroal.pool.util.UncheckedArrayList;
import io.agroal.pool.wrapper.ConnectionWrapper;
import java.sql.Connection;
import java.sql.SQLException;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;

public final class ConnectionPool
implements AutoCloseable {
    private final AgroalConnectionPoolConfiguration configuration;
    private final DataSource dataSource;
    private final ThreadLocal<UncheckedArrayList<ConnectionHandler>> localCache;
    private final StampedCopyOnWriteArrayList<ConnectionHandler> allConnections;
    private final AgroalSynchronizer synchronizer = new AgroalSynchronizer();
    private final ConnectionFactory connectionFactory;
    private final PriorityScheduledExecutor housekeepingExecutor;
    private final TransactionIntegration transactionIntegration;
    private final boolean leakEnabled;
    private final boolean validationEnabled;
    private final boolean reapEnabled;
    private volatile long maxUsed = 0L;

    public ConnectionPool(AgroalConnectionPoolConfiguration configuration, DataSource dataSource) {
        this.configuration = configuration;
        this.dataSource = dataSource;
        this.allConnections = new StampedCopyOnWriteArrayList<ConnectionHandler>(ConnectionHandler.class);
        this.localCache = ThreadLocal.withInitial(() -> new UncheckedArrayList(ConnectionHandler.class));
        this.connectionFactory = new ConnectionFactory(configuration.connectionFactoryConfiguration());
        this.housekeepingExecutor = new PriorityScheduledExecutor(1, "Agroal_" + System.identityHashCode(this));
        this.transactionIntegration = configuration.transactionIntegration();
        this.leakEnabled = !configuration.leakTimeout().isZero();
        this.validationEnabled = !configuration.validationTimeout().isZero();
        this.reapEnabled = !configuration.reapTimeout().isZero();
    }

    public void init() {
        this.fill(this.configuration.initialSize());
        if (this.leakEnabled) {
            this.housekeepingExecutor.schedule(new LeakTask(), this.configuration.leakTimeout().toNanos(), TimeUnit.NANOSECONDS);
        }
        if (this.validationEnabled) {
            this.housekeepingExecutor.schedule(new ValidationTask(), this.configuration.validationTimeout().toNanos(), TimeUnit.NANOSECONDS);
        }
        if (this.reapEnabled) {
            this.housekeepingExecutor.schedule(new ReapTask(), this.configuration.reapTimeout().toNanos(), TimeUnit.NANOSECONDS);
        }
    }

    private void fill(int newSize) {
        int connectionCount = newSize - this.allConnections.size();
        while (connectionCount-- > 0) {
            this.newConnectionHandler();
        }
    }

    @Override
    public void close() {
        this.housekeepingExecutor.shutdownNow();
    }

    private Future<?> newConnectionHandler() {
        return this.housekeepingExecutor.executeNow(() -> {
            if (this.allConnections.size() >= this.configuration.maxSize()) {
                return;
            }
            ListenerHelper.fireBeforeConnectionCreation(this.dataSource);
            long metricsStamp = this.dataSource.metricsRepository().beforeConnectionCreation();
            try {
                ConnectionHandler handler = new ConnectionHandler(this.connectionFactory.createConnection(), this);
                handler.setState(ConnectionHandler.State.CHECKED_IN);
                this.allConnections.add(handler);
                this.maxUsed = Math.max(this.maxUsed, (long)this.allConnections.size());
                this.dataSource.metricsRepository().afterConnectionCreation(metricsStamp);
                ListenerHelper.fireOnConnectionCreation(this.dataSource, handler);
            }
            catch (SQLException e) {
                throw new RuntimeException("Exception while creating new connection", e);
            }
            finally {
                this.synchronizer.releaseConditional();
            }
        });
    }

    public Connection getConnection() throws SQLException {
        ListenerHelper.fireBeforeConnectionAcquire(this.dataSource);
        long metricsStamp = this.dataSource.metricsRepository().beforeConnectionAcquire();
        if (this.housekeepingExecutor.isShutdown()) {
            throw new SQLException("This pool is closed and does not handle any more connections!");
        }
        ConnectionHandler checkedOutHandler = null;
        ConnectionWrapper connectionWrapper = this.wrapperFromTransaction();
        if (connectionWrapper != null) {
            checkedOutHandler = connectionWrapper.getHandler();
        }
        if (checkedOutHandler == null) {
            checkedOutHandler = this.handlerFromLocalCache();
        }
        if (checkedOutHandler == null) {
            checkedOutHandler = this.handlerFromSharedCache();
        }
        this.dataSource.metricsRepository().afterConnectionAcquire(metricsStamp);
        ListenerHelper.fireOnConnectionAcquired(this.dataSource, checkedOutHandler);
        if (this.leakEnabled || this.reapEnabled) {
            checkedOutHandler.setLastAccess(System.nanoTime());
        }
        if (this.leakEnabled) {
            if (checkedOutHandler.getHoldingThread() != null) {
                Throwable warn = new Throwable("Shared connection between threads '" + checkedOutHandler.getHoldingThread().getName() + "' and '" + Thread.currentThread().getName() + "'");
                warn.setStackTrace(checkedOutHandler.getHoldingThread().getStackTrace());
                ListenerHelper.fireOnWarning(this.dataSource, warn);
            }
            checkedOutHandler.setHoldingThread(Thread.currentThread());
        }
        connectionWrapper = new ConnectionWrapper(checkedOutHandler);
        this.transactionIntegration.associate((Connection)connectionWrapper);
        return connectionWrapper;
    }

    private ConnectionWrapper wrapperFromTransaction() throws SQLException {
        return (ConnectionWrapper)this.transactionIntegration.getConnection();
    }

    private ConnectionHandler handlerFromLocalCache() {
        UncheckedArrayList<ConnectionHandler> cachedConnections = this.localCache.get();
        while (!cachedConnections.isEmpty()) {
            ConnectionHandler handler = cachedConnections.removeLast();
            if (!handler.setState(ConnectionHandler.State.CHECKED_IN, ConnectionHandler.State.CHECKED_OUT)) continue;
            return handler;
        }
        return null;
    }

    /*
     * Unable to fully structure code
     */
    private ConnectionHandler handlerFromSharedCache() throws SQLException {
        remaining = this.configuration.acquisitionTimeout().toNanos();
        remaining = remaining > 0L ? remaining : 0x7FFFFFFFFFFFFFFFL;
        try {
            while (true) lbl-1000:
            // 4 sources

            {
                for (ConnectionHandler handler : this.allConnections.getUnderlyingArray()) {
                    if (!handler.setState(ConnectionHandler.State.CHECKED_IN, ConnectionHandler.State.CHECKED_OUT)) continue;
                    return handler;
                }
                if (this.allConnections.size() < this.configuration.maxSize()) {
                    this.newConnectionHandler().get();
                    ** continue;
                }
                synchronizationStamp = this.synchronizer.getStamp();
                start = System.nanoTime();
                if (remaining < 0L || !this.synchronizer.tryAcquireNanos(synchronizationStamp, remaining)) {
                    throw new SQLException("Sorry, acquisition timeout!");
                }
                remaining -= System.nanoTime() - start;
            }
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new SQLException("Interrupted while acquiring");
        }
        catch (ExecutionException e) {
            throw new SQLException("Exception while creating new connection", e);
        }
    }

    public void returnConnection(ConnectionHandler handler) throws SQLException {
        ListenerHelper.fireBeforeConnectionReturn(this.dataSource, handler);
        if (this.leakEnabled) {
            handler.setHoldingThread(null);
        }
        if (this.reapEnabled) {
            handler.setLastAccess(System.nanoTime());
        }
        if (this.transactionIntegration.disassociate(handler.getConnection())) {
            handler.resetConnection(this.configuration.connectionFactoryConfiguration());
            this.localCache.get().add(handler);
            handler.setState(ConnectionHandler.State.CHECKED_IN);
            this.synchronizer.releaseConditional();
            this.dataSource.metricsRepository().afterConnectionReturn();
            ListenerHelper.fireOnConnectionReturn(this.dataSource, handler);
        }
    }

    private long activeCount(ConnectionHandler[] handlers) {
        int l = 0;
        for (ConnectionHandler handler : handlers) {
            if (!handler.isActive()) continue;
            ++l;
        }
        return l;
    }

    public long activeCount() {
        return this.activeCount(this.allConnections.getUnderlyingArray());
    }

    public long availableCount() {
        ConnectionHandler[] handlers = this.allConnections.getUnderlyingArray();
        return (long)handlers.length - this.activeCount(handlers);
    }

    public long maxUsedCount() {
        return this.maxUsed;
    }

    public void resetMaxUsedCount() {
        this.maxUsed = 0L;
    }

    public long awaitingCount() {
        return this.synchronizer.getQueueLength();
    }

    private final class DestroyConnectionTask
    implements Runnable {
        private final ConnectionHandler handler;

        public DestroyConnectionTask(ConnectionHandler handler) {
            this.handler = handler;
        }

        @Override
        public void run() {
            ListenerHelper.fireBeforeConnectionDestroy(ConnectionPool.this.dataSource, this.handler);
            try {
                this.handler.closeConnection();
            }
            catch (SQLException e) {
                ListenerHelper.fireOnWarning(ConnectionPool.this.dataSource, e);
            }
            this.handler.setState(ConnectionHandler.State.DESTROYED);
            ListenerHelper.fireOnConnectionDestroy(ConnectionPool.this.dataSource, this.handler);
        }
    }

    private final class ReapTask
    implements Runnable {
        private ReapTask() {
        }

        @Override
        public void run() {
            ConnectionPool.this.allConnections.forEach(hadler -> ConnectionPool.this.housekeepingExecutor.submit(new ReapConnectionTask((ConnectionHandler)hadler)));
            ConnectionPool.this.housekeepingExecutor.schedule(this, ConnectionPool.this.configuration.reapTimeout().toNanos(), TimeUnit.NANOSECONDS);
        }

        private class ReapConnectionTask
        implements Runnable {
            private final ConnectionHandler handler;

            public ReapConnectionTask(ConnectionHandler handler) {
                this.handler = handler;
            }

            @Override
            public void run() {
                ListenerHelper.fireBeforeConnectionReap(ConnectionPool.this.dataSource, this.handler);
                if (ConnectionPool.this.allConnections.size() > ConnectionPool.this.configuration.minSize() && this.handler.setState(ConnectionHandler.State.CHECKED_IN, ConnectionHandler.State.FLUSH)) {
                    if (System.nanoTime() - this.handler.getLastAccess() > ConnectionPool.this.configuration.reapTimeout().toNanos()) {
                        ConnectionPool.this.allConnections.remove(this.handler);
                        ConnectionPool.this.dataSource.metricsRepository().afterConnectionReap();
                        ListenerHelper.fireOnConnectionReap(ConnectionPool.this.dataSource, this.handler);
                        ConnectionPool.this.housekeepingExecutor.execute(new DestroyConnectionTask(this.handler));
                    } else {
                        this.handler.setState(ConnectionHandler.State.CHECKED_IN);
                    }
                }
            }
        }
    }

    private final class ValidationTask
    implements Runnable {
        private ValidationTask() {
        }

        @Override
        public void run() {
            ConnectionPool.this.allConnections.forEach(hadler -> ConnectionPool.this.housekeepingExecutor.submit(new ValidateConnectionTask((ConnectionHandler)hadler)));
            ConnectionPool.this.housekeepingExecutor.schedule(this, ConnectionPool.this.configuration.validationTimeout().toNanos(), TimeUnit.NANOSECONDS);
        }

        private class ValidateConnectionTask
        implements Runnable {
            private final ConnectionHandler handler;

            public ValidateConnectionTask(ConnectionHandler handler) {
                this.handler = handler;
            }

            @Override
            public void run() {
                ListenerHelper.fireBeforeConnectionValidation(ConnectionPool.this.dataSource, this.handler);
                if (this.handler.setState(ConnectionHandler.State.CHECKED_IN, ConnectionHandler.State.VALIDATION)) {
                    if (ConnectionPool.this.configuration.connectionValidator().isValid(this.handler.getConnection())) {
                        this.handler.setState(ConnectionHandler.State.CHECKED_IN);
                        ListenerHelper.fireOnConnectionValid(ConnectionPool.this.dataSource, this.handler);
                    } else {
                        this.handler.setState(ConnectionHandler.State.FLUSH);
                        ConnectionPool.this.allConnections.remove(this.handler);
                        ConnectionPool.this.dataSource.metricsRepository().afterConnectionInvalid();
                        ListenerHelper.fireOnConnectionInvalid(ConnectionPool.this.dataSource, this.handler);
                        ConnectionPool.this.housekeepingExecutor.execute(new DestroyConnectionTask(this.handler));
                    }
                }
            }
        }
    }

    private final class LeakTask
    implements Runnable {
        private LeakTask() {
        }

        @Override
        public void run() {
            for (ConnectionHandler handler : (ConnectionHandler[])ConnectionPool.this.allConnections.getUnderlyingArray()) {
                ConnectionPool.this.housekeepingExecutor.execute(new LeakConnectionTask(handler));
            }
            ConnectionPool.this.housekeepingExecutor.schedule(this, ConnectionPool.this.configuration.leakTimeout().toNanos(), TimeUnit.NANOSECONDS);
        }

        private class LeakConnectionTask
        implements Runnable {
            private final ConnectionHandler handler;

            public LeakConnectionTask(ConnectionHandler handler) {
                this.handler = handler;
            }

            @Override
            public void run() {
                ListenerHelper.fireBeforeConnectionLeak(ConnectionPool.this.dataSource, this.handler);
                Thread thread = this.handler.getHoldingThread();
                if (thread != null && System.nanoTime() - this.handler.getLastAccess() > ConnectionPool.this.configuration.leakTimeout().toNanos()) {
                    ConnectionPool.this.dataSource.metricsRepository().afterLeakDetection();
                    ListenerHelper.fireOnConnectionLeak(ConnectionPool.this.dataSource, this.handler);
                }
            }
        }
    }
}

