/*
 * Licensed to the University Corporation for Advanced Internet Development, 
 * Inc. (UCAID) under one or more contributor license agreements.  See the 
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache 
 * License, Version 2.0 (the "License"); you may not use this file except in 
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.saml.attribute.resolver.impl;

import java.util.Collections;
import java.util.List;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import net.shibboleth.idp.attribute.IdPAttribute;
import net.shibboleth.idp.attribute.IdPAttributeValue;
import net.shibboleth.idp.attribute.StringAttributeValue;
import net.shibboleth.idp.attribute.resolver.AbstractDataConnector;
import net.shibboleth.idp.attribute.resolver.PluginDependencySupport;
import net.shibboleth.idp.attribute.resolver.ResolverPluginDependency;
import net.shibboleth.idp.attribute.resolver.context.AttributeResolverWorkContext;
import net.shibboleth.utilities.java.support.annotation.constraint.NonnullAfterInit;
import net.shibboleth.utilities.java.support.component.ComponentInitializationException;
import net.shibboleth.utilities.java.support.component.ComponentSupport;
import net.shibboleth.utilities.java.support.primitive.StringSupport;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * The basis of a {@link net.shibboleth.idp.attribute.resolver.DataConnector} that handles persistent IDs that depend on
 * a source {@link IdPAttribute}.
 */
public abstract class AbstractPersistentIdDataConnector extends AbstractDataConnector {

    /** Class logger. */
    @Nonnull private final Logger log = LoggerFactory.getLogger(AbstractPersistentIdDataConnector.class);

    /** ID of the attribute generated by this data connector. */
    @NonnullAfterInit private String generatedAttribute;

    /** ID of the attribute whose first value is used when generating the computed ID. */
    @NonnullAfterInit private String sourceAttribute;

    /**
     * Get the ID of the attribute whose first value is used when generating the computed ID.
     * 
     * @return ID of the attribute whose first value is used when generating the computed ID
     */
    @NonnullAfterInit public String getSourceAttributeId() {
        return sourceAttribute;
    }

    /**
     * Set the ID of the attribute whose first value is used when generating the computed ID.
     * 
     * @param newAttributeId what to set.
     */
    public void setSourceAttributeId(@Nullable String newAttributeId) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        sourceAttribute = StringSupport.trimOrNull(newAttributeId);
    }

    /**
     * Get the ID of the attribute generated by this connector.
     * 
     * @return ID of the attribute generated by this connector
     */
    @NonnullAfterInit public String getGeneratedAttributeId() {
        return generatedAttribute;
    }

    /**
     * Set the ID of the attribute generated by this connector.
     * 
     * @param newAttributeId what to set.
     */
    public void setGeneratedAttributeId(@Nullable String newAttributeId) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        generatedAttribute = newAttributeId;
    }

    /** {@inheritDoc} */
    @Override protected void doInitialize() throws ComponentInitializationException {

        if (null == getSourceAttributeId()) {
            throw new ComponentInitializationException(getLogPrefix() + " No source attribute present.");
        }

        // We have an input id, so that gets added to the dependencies.
        if (null != getSourceAttributeId()) {
            for (ResolverPluginDependency depends : getDependencies()) {
                depends.setDependencyAttributeId(getSourceAttributeId());
            }
        }
        super.doInitialize();

        if (null == generatedAttribute) {
            generatedAttribute = getId();
            log.info("{} No generated attribute ID supplied, using ID of connector: {}", getLogPrefix(),
                    generatedAttribute);
        }
    }

    /**
     * Helper function to locate the source Attribute in the dependencies.
     * 
     * @param workContext the context to look in
     * @return the value, or null in any of the failure cases.
     */
    @Nullable protected String resolveSourceAttribute(@Nonnull final AttributeResolverWorkContext workContext) {
        ComponentSupport.ifNotInitializedThrowUninitializedComponentException(this);
        ComponentSupport.ifNotInitializedThrowUninitializedComponentException(this);
        ComponentSupport.ifDestroyedThrowDestroyedComponentException(this);

        final List<IdPAttributeValue<?>> attributeValues =
                PluginDependencySupport.getMergedAttributeValues(workContext, getDependencies());
        if (attributeValues == null || attributeValues.isEmpty()) {
            log.debug("{} Source attribute {} for connector {} provide no values", getLogPrefix(),
                    getSourceAttributeId(), getId());
            return null;
        }

        if (attributeValues.size() > 1) {
            log.warn("{} Source attribute {} for connector {} has more than one value, only one value is used",
                    getLogPrefix(), getSourceAttributeId(), getId());
        }

        final IdPAttributeValue attributeValue = attributeValues.iterator().next();

        final String val;

        if (attributeValue instanceof StringAttributeValue) {
            if (StringSupport.trimOrNull((String) attributeValue.getValue()) == null) {
                log.warn("{} Source attribute {} for connector {} was all-whitespace", getLogPrefix(),
                        getSourceAttributeId(), getId());
                return null;
            }
            val = (String) attributeValue.getValue();
        } else {
            log.warn("{} Source attribute {} for connector {} was of an unsupported type: {}", getLogPrefix(),
                    getSourceAttributeId(), getId(), attributeValue.getClass().getName());
            return null;
        }

        if (val == null) {
            log.warn("{} Attribute value {} for connector {} resolved as empty or null", getLogPrefix(),
                    getSourceAttributeId(), getId());
        }
        return val;
    }

    /**
     * Encode the provided string.
     * 
     * @param value the value to encode or null if that failed
     * @return null or the attribute.
     */
    @Nullable protected Map<String, IdPAttribute> encodeAsAttribute(@Nullable String value) {
        ComponentSupport.ifNotInitializedThrowUninitializedComponentException(this);
        if (null == value) {
            // The message will have been logged above
            return null;
        }
        final IdPAttribute attribute = new IdPAttribute(getGeneratedAttributeId());
        attribute.setValues(Collections.singletonList(StringAttributeValue.valueOf(value)));
        return Collections.singletonMap(getGeneratedAttributeId(), attribute);
    }

}