/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.oidc.op.messaging;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.nimbusds.common.contenttype.ContentType;
import com.nimbusds.oauth2.sdk.ErrorObject;
import com.nimbusds.oauth2.sdk.ErrorResponse;
import com.nimbusds.oauth2.sdk.http.HTTPResponse;

import net.minidev.json.JSONObject;
import net.shibboleth.utilities.java.support.logic.Constraint;

/** Class for creating JSON Error response for requests expecting JSON response. */
public class JSONErrorResponse implements ErrorResponse {

    /** Error object. */
    private ErrorObject error;

    /** cache control value. */
    private String cacheControl;

    /** pragma value. */
    private String pragma;

    /**
     * Constructor.
     * 
     * @param errorObject error. MUST not be null.
     */
    public JSONErrorResponse(@Nonnull final ErrorObject errorObject) {
        this(errorObject, null, null);
    }

    /**
     * Constructor.
     * 
     * @param errorObject JSON content.
     * @param cacheControlValue cache control value.
     * @param pragmaValue pragma value.
     */
    public JSONErrorResponse(@Nonnull final ErrorObject errorObject, @Nullable final String cacheControlValue,
            @Nullable final String pragmaValue) {
        Constraint.isNotNull(errorObject, "content cannot be null");
        error = errorObject;
        cacheControl = cacheControlValue;
        pragma = pragmaValue;
    }

    /** {@inheritDoc} */
    @Override
    public boolean indicatesSuccess() {
        return true;
    }

    /**
     * Error content as json.
     * 
     * @return error as json.
     */
    private String getContent() {
        final JSONObject content = new JSONObject();
        if (error == null) {
            return null;
        }
        content.put("error", error.getCode());
        if (error.getDescription() != null) {
            content.put("error_description", error.getDescription());
        }
        if (error.getURI() != null) {
            content.put("error_uri", error.getURI().toString());
        }
        return content.toString();
    }

    /** {@inheritDoc} */
    @Override
    public HTTPResponse toHTTPResponse() {
        final HTTPResponse httpResponse = new HTTPResponse(error.getHTTPStatusCode());
        httpResponse.setEntityContentType(ContentType.APPLICATION_JSON);
        if (cacheControl != null) {
            httpResponse.setCacheControl(cacheControl);
        }
        if (pragma != null) {
            httpResponse.setPragma(pragma);
        }
        httpResponse.setContent(getContent());
        return httpResponse;
    }

    /** {@inheritDoc} */
    @Override
    public ErrorObject getErrorObject() {
        return error;
    }
}
