/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.oidc.op.messaging;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.nimbusds.common.contenttype.ContentType;
import com.nimbusds.oauth2.sdk.SuccessResponse;
import com.nimbusds.oauth2.sdk.http.HTTPResponse;

import net.minidev.json.JSONObject;
import net.shibboleth.utilities.java.support.logic.Constraint;

/** Class for creating JSON Success response. */
public class JSONSuccessResponse implements SuccessResponse {

    /** JSON content. */
    private JSONObject content;

    /** cache control value. */
    private String cacheControl;

    /** pragma value. */
    private String pragma;

    /**
     * Constructor.
     * 
     * @param contentObject JSON content. MUST not be null.
     */
    public JSONSuccessResponse(@Nonnull final JSONObject contentObject) {
        this(contentObject, null, null);

    }

    /**
     * Constructor.
     * 
     * @param contentObject JSON content
     * @param cacheControlValue cache control value
     * @param pragmaValue pragma value
     */
    public JSONSuccessResponse(@Nonnull final JSONObject contentObject, @Nullable final String cacheControlValue,
            @Nullable final String pragmaValue) {
        Constraint.isNotNull(contentObject, "content cannot be null");
        content = contentObject;
        cacheControl = cacheControlValue;
        pragma = pragmaValue;
    }

    /** {@inheritDoc} */
    @Override
    public boolean indicatesSuccess() {
        return true;
    }

    /** {@inheritDoc} */
    @Override
    public HTTPResponse toHTTPResponse() {
        final HTTPResponse httpResponse = new HTTPResponse(HTTPResponse.SC_OK);
        httpResponse.setEntityContentType(ContentType.APPLICATION_JSON);
        if (cacheControl != null) {
            httpResponse.setCacheControl(cacheControl);
        }
        if (pragma != null) {
            httpResponse.setPragma(pragma);
        }
        httpResponse.setContent(content.toJSONString());
        return httpResponse;
    }
}
