/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.oidc.op.messaging.context;

import java.net.URI;
import java.time.Duration;
import java.time.Instant;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.opensaml.messaging.context.BaseContext;
import com.nimbusds.jwt.JWT;
import com.nimbusds.oauth2.sdk.AuthorizationCode;
import com.nimbusds.oauth2.sdk.Scope;
import com.nimbusds.oauth2.sdk.token.AccessToken;
import com.nimbusds.oauth2.sdk.token.BearerAccessToken;
import com.nimbusds.oauth2.sdk.token.RefreshToken;
import com.nimbusds.openid.connect.sdk.OIDCClaimsRequest;
import com.nimbusds.openid.connect.sdk.claims.ACR;
import com.nimbusds.openid.connect.sdk.claims.IDTokenClaimsSet;
import com.nimbusds.openid.connect.sdk.claims.UserInfo;

import net.shibboleth.idp.attribute.AttributesMapContainer;
import net.shibboleth.idp.plugin.oidc.op.token.support.TokenClaimsSet;

/**
 * Subcontext carrying information to form authentication, token and userinfo responses for relying party. This context
 * appears as a subcontext of the {@link org.opensaml.messaging.context.MessageContext}.
 */
public class OIDCAuthenticationResponseContext extends BaseContext {

    /** The id token formed. */
    @Nullable
    private IDTokenClaimsSet idToken;

    /** The request object. */
    @Nullable
    private JWT requestObject;

    /** The user info formed. */
    @Nullable
    private UserInfo userInfo;

    /** The signed/encrypted id token / user info response formed. */
    @Nullable
    private JWT processedToken;

    /** the acr used in response. **/
    @Nullable
    private ACR acr;

    /** validated redirect uri. */
    @Nullable
    private URI redirectURI;

    /** Authentication time of the end user. */
    @Nullable
    private Instant authTime;

    /** Validated scope values. */
    @Nullable
    private Scope requestedScope;

    /** Requested sub value. */
    @Nullable
    private String requestedSubject;

    /** Subject generated for response. Value is set to sub claim. */
    @Nullable
    private String subject;

    /** Subject type, public or pairwise. */
    @Nullable
    private String subjectType;

    /** Authorization code. */
    @Nullable
    private AuthorizationCode authorizationCode;

    /** Access token. */
    @Nullable
    private AccessToken accessToken;

    /** Refresh token. */
    @Nullable
    private RefreshToken refreshToken;

    /** Token (authz code, access token) claims. */
    @Nullable
    private TokenClaimsSet tokenClaims;

    /** Requested claims. */
    @Nullable
    private OIDCClaimsRequest requestedClaims;
    
    /** Mapped requested claims from the ID Token set. */
    @Nullable private AttributesMapContainer mappedIdTokenRequestedClaims;

    /** Mapped requested claims from the Userinfo set. */
    @Nullable private AttributesMapContainer mappedUserinfoRequestedClaims;
    
    /**
     * Get requested claims.
     * 
     * @return requested claims
     */
    @Nullable
    public OIDCClaimsRequest getRequestedClaims() {
        return requestedClaims;
    }

    /**
     * Set requested claims.
     * 
     * @param claims requested claims
     */
    public void setRequestedClaims(@Nullable final OIDCClaimsRequest claims) {
        requestedClaims = claims;
    }
    
    /**
     * Get the requested claims for the ID Token after they've been reverse-mapped by the registry.
     * 
     * @return mapped requested claims
     */
    @Nullable public AttributesMapContainer getMappedIdTokenRequestedClaims() {
        return mappedIdTokenRequestedClaims;
    }
    
    /**
     * Set the requested claims for the ID Token after they've been reverse-mapped by the registry.
     * 
     * @param container the container of mapped claims
     */
    public void setMappedIdTokenRequestedClaims(@Nullable final AttributesMapContainer container) {
        mappedIdTokenRequestedClaims = container;
    }

    /**
     * Get the requested claims for the Userinfo endpoint after they've been reverse-mapped by the registry.
     * 
     * @return mapped requested claims
     */
    @Nullable public AttributesMapContainer getMappedUserinfoRequestedClaims() {
        return mappedUserinfoRequestedClaims;
    }

    /**
     * Set the requested claims for the Userinfo endpoint after they've been reverse-mapped by the registry.
     * 
     * @param container the container of mapped claims
     */
    public void setMappedUserinfoRequestedClaims(@Nullable final AttributesMapContainer container) {
        mappedUserinfoRequestedClaims = container;
    }
    
    /**
     * Get token claims.
     * 
     * @return token claims
     */
    @Nullable
    public TokenClaimsSet getTokenClaimsSet() {
        return tokenClaims;
    }

    /**
     * Set token claims.
     * 
     * @param claims token claims
     */
    public void setTokenClaimsSet(@Nullable final TokenClaimsSet claims) {
        tokenClaims = claims;
    }

    /**
     * Get authorization code.
     * 
     * @return authorization code
     */
    @Nullable
    public AuthorizationCode getAuthorizationCode() {
        return authorizationCode;
    }

    /**
     * Set authorization code.
     * 
     * @param code String to form authorization code
     */
    public void setAuthorizationCode(@Nullable final String code) {
        authorizationCode = code == null ? null : new AuthorizationCode(code);
    }

    /**
     * Get access token.
     * 
     * @return access token
     */
    @Nullable
    public AccessToken getAccessToken() {
        return accessToken;
    }

    /**
     * Set access token.
     * 
     * @param token String to form access token
     * @param lifeTime lifetime of the access token
     */
    public void setAccessToken(@Nullable final String token, @Nonnull final Duration lifeTime) {
        accessToken = token == null ? null : new BearerAccessToken(token, lifeTime.getSeconds(), null);
    }

    /**
     * Get refresh token.
     * 
     * @return refresh token
     */
    @Nullable
    public RefreshToken getRefreshToken() {
        return refreshToken;
    }

    /**
     * Set refresh token.
     * 
     * @param token String to form refresh token
     */
    public void setRefreshToken(@Nullable final String token) {
        refreshToken = token == null ? null : new RefreshToken(token);
    }

    /**
     * Gets requested sub value.
     * 
     * @return requested sub value
     */
    @Nullable
    public String getRequestedSubject() {
        return requestedSubject;
    }

    /**
     * Set requested sub value.
     * 
     * @param sub requested sub value.
     */
    public void setRequestedSubject(@Nullable final String sub) {
        this.requestedSubject = sub;
    }

    /**
     * Gets Name ID generated for response.
     * 
     * @return Name ID generated for response
     */
    @Nullable
    public String getSubject() {
        return subject;
    }

    /**
     * Sets generated subject for the response.
     * 
     * @param generatedSubject subject for the response 
     */
    public void setSubject(@Nullable final String generatedSubject) {
        subject = generatedSubject;
    }

    /**
     * Gets subject type.
     * 
     * @return generated subject type.
     */
    @Nullable
    public String getSubjectType() {
        return subjectType;
    }

    /**
     * Sets subject type.
     * 
     * @param type subject type.
     */
    public void setSubjectType(@Nullable final String type) {
        subjectType = type;
    }

    /**
     * Get validated scope values.
     * 
     * @return validated scope values
     */
    @Nullable
    public Scope getScope() {
        return requestedScope;
    }

    /**
     * Set validated scope values.
     * 
     * @param scope scope values
     */
    public void setScope(@Nullable final Scope scope) {
        requestedScope = scope;
    }

    /**
     * Authentication time of the end user.
     * 
     * @return authentication time of the end user. null if has not been set.
     */
    @Nullable
    public Instant getAuthTime() {
        return authTime;
    }

    /**
     * Set authentication time of the end user.
     * 
     * @param time authentication time.
     */
    public void setAuthTime(@Nullable final Instant time) {
        authTime = time;
    }

    /**
     * Returns a validated redirect uri for the response.
     * 
     * @return redirect uri.
     */
    @Nullable
    public URI getRedirectURI() {
        return redirectURI;
    }

    /**
     * Sets a validated redirect uri for the response.
     * 
     * @param uri validated redirect uri for the response
     */
    public void setRedirectURI(@Nullable final URI uri) {
        redirectURI = uri;
    }

    /**
     * Returns the acr meant for response.
     * 
     * @return acr
     */
    @Nullable
    public ACR getAcr() {
        return acr;
    }

    /**
     * Set acr for response.
     * 
     * @param acrValue for response.
     */
    public void setAcr(@Nullable final String acrValue) {
        if (acrValue != null) {
            acr = new ACR(acrValue);

        } else {
            acr = null;
        }
    }

    /**
     * Get the id token.
     * 
     * @return The id token.
     */
    @Nullable
    public IDTokenClaimsSet getIDToken() {
        return idToken;
    }

    /**
     * Set the id token.
     * 
     * @param token The id token.
     */
    public void setIDToken(@Nullable final IDTokenClaimsSet token) {
        idToken = token;
    }

    /**
     * Get the user info.
     * 
     * @return The user info.
     */
    @Nullable
    public UserInfo getUserInfo() {
        return userInfo;
    }

    /**
     * Set the user info.
     * 
     * @param info The user info.
     */
    public void setUserInfo(@Nullable final UserInfo info) {
        userInfo = info;
    }

    /**
     * Get the signed/encrypted id token / user info response.
     * 
     * @return The signed id token / user info response
     */
    @Nullable
    public JWT getProcessedToken() {
        return processedToken;
    }

    /**
     * Set the signed/encrypted id token / user info response.
     * 
     * @param token The signed id token / user info response
     */
    public void setProcessedToken(@Nullable final JWT token) {
        processedToken = token;
    }

    /**
     * Get the request object.
     * 
     * @return the request object
     */
    @Nullable
    public JWT getRequestObject() {
        return requestObject;
    }

    /**
     * Set the request object. Either by value or fetched by reference.
     * 
     * @param obj the request object.
     */
    public void setRequestObject(@Nullable final JWT obj) {
        requestObject = obj;
    }
}