/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.oidc.op.profile.context.navigate;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.opensaml.messaging.context.navigate.ContextDataLookupFunction;
import org.opensaml.profile.context.ProfileRequestContext;

import com.nimbusds.jwt.JWT;
import com.nimbusds.openid.connect.sdk.AuthenticationRequest;

import net.shibboleth.idp.plugin.oidc.op.messaging.context.OIDCAuthenticationResponseContext;

/**
 * A Abstract function extended by lookups searching fields from authentication request.
 * 
 * @param <T> type of lookup result to return.
 */
public abstract class AbstractAuthenticationRequestLookupFunction<T>
        implements ContextDataLookupFunction<ProfileRequestContext,T> {

    /** Request. */
    @Nullable private JWT requestObject;


    /**
     * Implemented to perform the actual lookup.
     * 
     * @param req authentication request to perform the lookup from.
     * @return lookup value.
     */
    @Nullable protected abstract T doLookup(@Nonnull AuthenticationRequest req);

    /** {@inheritDoc} */
    @Nullable
    public T apply(@Nullable final ProfileRequestContext input) {
        if (input == null || input.getInboundMessageContext() == null) {
            return null;
        }
        final Object message = input.getInboundMessageContext().getMessage();
        if (message == null || !(message instanceof AuthenticationRequest)) {
            return null;
        }
        if (input.getOutboundMessageContext() != null) {
            final OIDCAuthenticationResponseContext ctx =
                    input.getOutboundMessageContext().getSubcontext(OIDCAuthenticationResponseContext.class, false);
            if (ctx != null) {
                requestObject = ctx.getRequestObject();
            }
        }
        
        return doLookup((AuthenticationRequest) message);
    }

    /**
     * Gets the request object.
     * 
     * @return the request object.
     */
    @Nullable public JWT getRequestObject() {
        return requestObject;
    }
    
}