/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.oidc.op.profile.context.navigate;

import java.text.ParseException;
import javax.annotation.Nonnull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.nimbusds.openid.connect.sdk.AuthenticationRequest;
import com.nimbusds.openid.connect.sdk.Nonce;

/**
 * A function that returns copy of Nonce via a lookup function. This default lookup locates nonce from oidc
 * authentication request if available. If information is not available, null is returned. If there is nonce in request
 * object it is used instead of nonce parameter.
 */
public class DefaultRequestNonceLookupFunction extends AbstractAuthenticationRequestLookupFunction<Nonce> {

    /** Class logger. */
    @Nonnull
    private Logger log = LoggerFactory.getLogger(DefaultRequestLoginHintLookupFunction.class);

    /** {@inheritDoc} */
    @Override
    protected  Nonce doLookup(@Nonnull final AuthenticationRequest req) {
        try {
            if (getRequestObject() != null && getRequestObject().getJWTClaimsSet().getClaim("nonce") != null) {
                final Object nonce = getRequestObject().getJWTClaimsSet().getClaim("nonce");
                if (nonce instanceof String) {
                    return new Nonce((String) nonce);
                } else {
                    log.error("nonce claim is not of expected type");
                    return null;
                }
            }
        } catch (final ParseException e) {
            log.error("Unable to parse nonce from request object nonce value");
            return null;
        }
        if (req.getNonce() != null) {
            return new Nonce(req.getNonce().getValue());
        }
        return null;
    }
}