/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.oidc.op.profile.context.navigate;

import java.net.URI;
import java.net.URISyntaxException;
import java.text.ParseException;
import javax.annotation.Nonnull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.nimbusds.openid.connect.sdk.AuthenticationRequest;

/**
 * A function that returns redirect uri of the authentication request via a lookup function. This default lookup locates
 * redirect uri from oidc authentication request if available. If information is not available, null is returned. If
 * there is redirect uri in request object it is used instead of redirect_uri parameter.
 */
public class DefaultRequestRedirectURILookupFunction extends AbstractAuthenticationRequestLookupFunction<URI> {

    /** Class logger. */
    @Nonnull
    private Logger log = LoggerFactory.getLogger(DefaultRequestRedirectURILookupFunction.class);
    
    /** {@inheritDoc} */
    @Override
    protected URI doLookup(@Nonnull final AuthenticationRequest req) {
        try {
            if (getRequestObject() != null && getRequestObject().getJWTClaimsSet().getClaim("redirect_uri") != null) {
                final Object redirect = getRequestObject().getJWTClaimsSet().getClaim("redirect_uri");
                if (redirect instanceof String) {
                    return new URI((String)redirect);
                } else {
                    log.error("login_hint claim is not of expected type");
                    return null;
                }
            }
        } catch (final ParseException | URISyntaxException e) {
            log.error("Unable to parse login hint from request object login_hint value");
            return null;
        }
        return req.getRedirectionURI();
    }
}