/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.oidc.op.profile.context.navigate;

import java.net.URI;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import net.shibboleth.idp.plugin.oidc.op.messaging.context.OIDCMetadataContext;
import net.shibboleth.utilities.java.support.component.AbstractIdentifiableInitializableComponent;
import net.shibboleth.utilities.java.support.logic.Constraint;

import org.opensaml.messaging.context.navigate.ContextDataLookupFunction;
import org.opensaml.profile.context.ProfileRequestContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * A function that returns sector identifier obtained via a lookup function.
 * 
 * The value is host component of registered sector_identifier_uri or host component of the registered redirect_uri in
 * case there is no sector_identifier_uri. In the latter case there must be only on registered redirect_uri or null is
 * returned.
 * 
 * <p>
 * If a specific setting is unavailable, a null value is returned.
 * </p>
 */
public class SectorIdentifierLookupFunction extends AbstractIdentifiableInitializableComponent
        implements ContextDataLookupFunction<ProfileRequestContext, String> {

    /** Class logger. */
    @Nonnull
    private final Logger log = LoggerFactory.getLogger(SectorIdentifierLookupFunction.class);

    /** Strategy function to lookup OIDC metadata context . */
    @Nonnull
    private Function<ProfileRequestContext, OIDCMetadataContext> oidcMetadataContextLookupStrategy;

    /**
     * Constructor.
     */
    public SectorIdentifierLookupFunction() {
        oidcMetadataContextLookupStrategy = new DefaultOIDCMetadataContextLookupFunction();
    }

    /**
     * Set the lookup strategy to use to locate the {@link OIDCMetadataContext}.
     * 
     * @param strategy lookup function to use
     */
    public void setOIDCMetadataContextLookupStrategy(
            @Nonnull final Function<ProfileRequestContext, OIDCMetadataContext> strategy) {
        oidcMetadataContextLookupStrategy =
                Constraint.isNotNull(strategy, "OIDCMetadata lookup strategy cannot be null");
    }

    /** {@inheritDoc} */
    @Nullable
    public String apply(@Nullable final ProfileRequestContext input) {
        if (input == null) {
            return null;
        }
        final OIDCMetadataContext ctx = oidcMetadataContextLookupStrategy.apply(input);
        if (ctx == null || ctx.getClientInformation() == null || ctx.getClientInformation().getOIDCMetadata() == null) {
            log.warn("OIDC metadata not available, no sector identifier can be resolved");
            return null;
        } else if (ctx.getClientInformation().getOIDCMetadata().getSectorIDURI() != null) {
            final String sectorIdentifier = ctx.getClientInformation().getOIDCMetadata().getSectorIDURI().getHost();
            log.debug("Sector identifier resolved by registered sector uri {}", sectorIdentifier);
            return sectorIdentifier;
        } else if (ctx.getClientInformation().getOIDCMetadata().getRedirectionURIs() == null) {
            log.warn("Redirection uri not available, unable to determine sector identifier");
            return null;
        }
        final Set<String> hosts = ctx.getClientInformation().getOIDCMetadata().getRedirectionURIs().stream()
                .map(URI::getHost).collect(Collectors.toSet());
        if (hosts.size() > 1) {
            log.warn("Multiple hosts found from the redirection uris but no registered sector uri");
            return null;
        }
        final String sectorIdentifier = hosts.stream().findFirst().get();
        log.debug("Sector identifier by single host from redirect uris: {}", sectorIdentifier);
        return sectorIdentifier;
    }

}