/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.oidc.op.profile.context.navigate;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.opensaml.messaging.context.MessageContext;
import org.opensaml.messaging.context.navigate.ContextDataLookupFunction;
import org.opensaml.profile.context.ProfileRequestContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.nimbusds.oauth2.sdk.id.ClientID;

import net.shibboleth.idp.plugin.oidc.op.messaging.context.OIDCAuthenticationResponseContext;

/**
 * For UserInfo end point.
 * 
 * A function that returns client id of the request via a lookup function. This lookup locates client id from access
 * token used for user info request if available. If information is not available, null is returned.
 */
public class UserInfoRequestClientIDLookupFunction implements ContextDataLookupFunction<MessageContext, ClientID> {

    /** Logger. */
    @Nonnull
    private final Logger log = LoggerFactory.getLogger(UserInfoRequestClientIDLookupFunction.class);

    /** {@inheritDoc} */
    @Nullable
    public ClientID apply(@Nullable final MessageContext input) {
        if (input == null) {
            return null;
        }
        if (!(input.getParent() instanceof ProfileRequestContext)) {
            return null;
        }
        final MessageContext msgCtx = ((ProfileRequestContext) input.getParent()).getOutboundMessageContext();
        if (msgCtx == null) {
            return null;
        }
        final OIDCAuthenticationResponseContext ctx = msgCtx.getSubcontext(OIDCAuthenticationResponseContext.class);
        if (ctx == null || ctx.getTokenClaimsSet() == null) {
            return null;
        }
        return ctx.getTokenClaimsSet().getClientID();

    }
}