/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.oidc.op.token.support;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.nimbusds.jwt.JWTClaimsSet;
import com.nimbusds.oauth2.sdk.Scope;
import com.nimbusds.oauth2.sdk.id.ClientID;
import com.nimbusds.openid.connect.sdk.Nonce;
import com.nimbusds.openid.connect.sdk.OIDCClaimsRequest;
import com.nimbusds.openid.connect.sdk.claims.ACR;
import com.nimbusds.openid.connect.sdk.claims.ClaimsSet;

import net.shibboleth.utilities.java.support.security.DataSealer;
import net.shibboleth.utilities.java.support.security.DataSealerException;
import net.shibboleth.utilities.java.support.security.IdentifierGenerationStrategy;

import java.net.URI;
import java.text.ParseException;
import java.time.Instant;
import java.util.List;

/** Class wrapping claims set for access token. */
public final class AccessTokenClaimsSet extends TokenClaimsSet {

    /** Value of access token claims set type. */
    private static final String VALUE_TYPE_AT = "at";

    /** Class logger. */
    @Nonnull
    private Logger log = LoggerFactory.getLogger(AccessTokenClaimsSet.class);

    /**
     * Constructor for access token claims set when derived from authz code.
     * 
     * @param tokenClaimSet Authorize Code / Refresh Token this token is based on. Must not be NULL.
     * @param scope Validated and possibly reduced scope of the authentication request. Must not be NULL.
     * @param dlClaims Claims set for token delivery.
     * @param dlClaimsUI Claims set for token delivery, user info only.
     * @param iat Issue time of the token. Must not be NULL.
     * @param exp Expiration time of the token. Must not be NULL.
     * @throws RuntimeException if called with non allowed null parameters
     */
    // Checkstyle: ParameterNumber OFF
    public AccessTokenClaimsSet(@Nonnull final TokenClaimsSet tokenClaimSet, @Nonnull final Scope scope,
            @Nullable final ClaimsSet dlClaims, @Nullable final ClaimsSet dlClaimsUI, @Nonnull final Instant iat,
            @Nonnull final Instant exp) {
        super(VALUE_TYPE_AT, tokenClaimSet.getID(), tokenClaimSet.getClientID(),
                tokenClaimSet.getClaimsSet().getIssuer(), tokenClaimSet.getPrincipal(),
                tokenClaimSet.getClaimsSet().getSubject(),
                tokenClaimSet.getACR() == null ? null : new ACR(tokenClaimSet.getACR()), iat, exp,
                tokenClaimSet.getNonce(), tokenClaimSet.getAuthenticationTime(), tokenClaimSet.getRedirectURI(), scope,
                tokenClaimSet.getClaimsRequest(), dlClaims, null, dlClaimsUI, tokenClaimSet.getConsentedClaims(),
                null, tokenClaimSet.isConsentEnabled());
    }
    // Checkstyle: ParameterNumber ON

    /**
     * Constructor for access token claims set.
     * 
     * @param idGenerator Generator for pseudo unique identifier for the code. Must not be NULL.
     * @param clientID Client Id of the rp. Must not be NULL.
     * @param issuer OP issuer value. Must not be NULL.
     * @param userPrincipal User Principal of the authenticated user. Must not be NULL.
     * @param subject subject of the authenticated user. Must not be NULL
     * @param acr Authentication context class reference value of the authentication. May be NULL.
     * @param iat Issue time of the authorize code. Must not be NULL.
     * @param exp Expiration time of the authorize code. Must not be NULL.
     * @param nonce Nonce of the authentication request. May be NULL.
     * @param authTime Authentication time of the user. Must not be NULL.
     * @param redirectURI Validated redirect URI of the authentication request. Must not be NULL.
     * @param scope Scope of the authentication request. Must not be NULL.
     * @param claims Claims request of the authentication request. May be NULL.
     * @param dlClaims token delivery claims delivered both for id token and userinfo response. May be NULL.
     * @param dlClaimsUI token delivery claims delivered for userinfo response. May be NULL.
     * @param consentedClaims consented claims. May be NULL.
     * @param consentEnabled Whether consent has been enabled.
     * @throws RuntimeException if called with nonallowed null parameters
     */
    // Checkstyle: ParameterNumber OFF
    private AccessTokenClaimsSet(@Nonnull final IdentifierGenerationStrategy idGenerator,
            @Nonnull final ClientID clientID, @Nonnull final String issuer, @Nonnull final String userPrincipal,
            @Nonnull final String subject, @Nullable final ACR acr, @Nonnull final Instant iat,
            @Nonnull final Instant exp, @Nullable final Nonce nonce, @Nonnull final Instant authTime,
            @Nonnull final URI redirectURI, @Nonnull final Scope scope, @Nullable final OIDCClaimsRequest claims,
            @Nullable final ClaimsSet dlClaims, @Nullable final ClaimsSet dlClaimsUI,
            @Nullable final List<Object> consentedClaims, final boolean consentEnabled) {
        super(VALUE_TYPE_AT, idGenerator.generateIdentifier(), clientID, issuer, userPrincipal, subject, acr, iat, exp,
                nonce, authTime, redirectURI, scope, claims, dlClaims, null, dlClaimsUI, consentedClaims, null,
                consentEnabled);
    }
    // Checkstyle: ParameterNumber ON

    /**
     * Private constructor for the parser.
     * 
     * @param accessTokenClaimsSet access token claims set
     */
    private AccessTokenClaimsSet(final JWTClaimsSet accessTokenClaimsSet) {
        setClaimsSet(accessTokenClaimsSet);
    }

    /**
     * Parses access token from string (JSON).
     * 
     * @param accessTokenClaimsSet JSON String representation of the code
     * @return AccessTokenClaimsSet instance if parsing is successful.
     * @throws ParseException if parsing fails for example due to incompatible types.
     */
    public static AccessTokenClaimsSet parse(final String accessTokenClaimsSet) throws ParseException {
        final JWTClaimsSet atClaimsSet = JWTClaimsSet.parse(accessTokenClaimsSet);
        // Throws exception if parsing result is not expected one.
        verifyParsedClaims(VALUE_TYPE_AT, atClaimsSet);
        return new AccessTokenClaimsSet(atClaimsSet);
    }

    /**
     * Parses access token from sealed access token.
     * 
     * @param wrappedAccessToken wrapped access token
     * @param dataSealer sealer to unwrap the access token
     * @return access token claims set.
     * @throws ParseException is thrown if unwrapped access token is not understood
     * @throws DataSealerException is thrown if unwrapping fails
     */
    public static AccessTokenClaimsSet parse(@Nonnull final String wrappedAccessToken,
            @Nonnull final DataSealer dataSealer) throws ParseException, DataSealerException {
        return parse(dataSealer.unwrap(wrappedAccessToken));
    }

    /** Builder to create instance of AccessTokenClaimsSet. */
    public static class Builder extends TokenClaimsSet.Builder<AccessTokenClaimsSet> {

        /**
         * Constructor for access token builder.
         * 
         * @param idGenerator Generator for pseudo unique identifier for the code. Must not be NULL.
         * @param clientID Client Id of the rp. Must not be NULL.
         * @param issuer OP issuer value. Must not be NULL.
         * @param userPrincipal User Principal of the authenticated user. Must not be NULL.
         * @param subject subject of the authenticated user. Must not be NULL
         * @param issuedAt Issue time of the authorize code. Must not be NULL.
         * @param expiresAt Expiration time of the authorize code. Must not be NULL.
         * @param authenticationTime Authentication time of the user. Must not be NULL.
         * @param redirectURI Validated redirect URI of the authentication request. Must not be NULL.
         * @param scope Scope of the authentication request. Must not be NULL.
         */
        // Checkstyle: ParameterNumber OFF
        public Builder(@Nonnull final IdentifierGenerationStrategy idGenerator, @Nonnull final ClientID clientID,
                @Nonnull final String issuer, @Nonnull final String userPrincipal, @Nonnull final String subject,
                @Nonnull final Instant issuedAt, @Nonnull final Instant expiresAt,
                @Nonnull final Instant authenticationTime, @Nonnull final URI redirectURI, @Nonnull final Scope scope) {
            super(idGenerator, clientID, issuer, userPrincipal, subject, issuedAt, expiresAt, authenticationTime,
                    redirectURI, scope);
        }
        // Checkstyle: ParameterNumber ON

        /**
         * Builds AccessTokenClaimsSet.
         * 
         * @return AccessTokenClaimsSet instance.
         */
        public AccessTokenClaimsSet build() {
            return new AccessTokenClaimsSet(idGen, rpId, iss, usrPrincipal, sub, acr, iat, exp, nonce, authTime,
                    redirect, reqScope, claims, dlClaims, dlClaimsUI, cnsntdClaims, cnsntEnabled);
        }

    }

}
