/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.oidc.op.token.support;

import java.util.Date;
import java.util.List;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.nimbusds.jwt.JWTClaimsSet;
import com.nimbusds.oauth2.sdk.Scope;
import com.nimbusds.oauth2.sdk.id.ClientID;
import com.nimbusds.oauth2.sdk.util.JSONObjectUtils;
import com.nimbusds.openid.connect.sdk.Nonce;
import com.nimbusds.openid.connect.sdk.OIDCClaimsRequest;
import com.nimbusds.openid.connect.sdk.claims.ACR;
import com.nimbusds.openid.connect.sdk.claims.ClaimsSet;

import net.minidev.json.JSONObject;
import net.shibboleth.utilities.java.support.security.DataSealer;
import net.shibboleth.utilities.java.support.security.DataSealerException;
import net.shibboleth.utilities.java.support.security.IdentifierGenerationStrategy;

import java.net.URI;
import java.text.ParseException;
import java.time.Instant;

/**
 * Class to extend for token claims sets. Offers the base functionality to Authorize Code, Refresh Token and Access
 * Token.
 */
public class TokenClaimsSet {

    /** Identifier for the token. */
    public static final String KEY_AC_ID = "jti";

    /** Type of the token. */
    public static final String KEY_TYPE = "type";

    /** OP issuer. */
    public static final String KEY_ISSUER = "iss";

    /** User principal representing authenticated user. */
    public static final String KEY_USER_PRINCIPAL = "prncpl";

    /** Subject of the user. */
    public static final String KEY_SUBJECT = "sub";

    /** Client id of the rp the token is generated for. */
    public static final String KEY_CLIENTID = "clid";

    /** Expiration time of the token. */
    public static final String KEY_EXPIRATION_TIME = "exp";

    /** Issue time of the token. */
    public static final String KEY_ISSUED_AT = "iat";

    /** Authentication context class reference value of the performed authentication. */
    public static final String KEY_ACR = "acr";

    /** Nonce of the original authentication request. */
    public static final String KEY_NONCE = "nonce";

    /** Authentication time of the performed authentication. */
    public static final String KEY_AUTH_TIME = "auth_time";

    /** Redirect uri of the original authentication request. */
    public static final String KEY_REDIRECT_URI = "redirect_uri";

    /** Scope of the original authentication request. */
    public static final String KEY_SCOPE = "scope";

    /** Claims request of the original authentication request. */
    public static final String KEY_CLAIMS = "claims";

    /** Claims set for token delivery. */
    public static final String KEY_DELIVERY_CLAIMS = "dl_claims";

    /** Claims set for token delivery, id token only. */
    public static final String KEY_DELIVERY_CLAIMS_IDTOKEN = "dl_claims_id";

    /** Claims set for token delivery, user info only. */
    public static final String KEY_DELIVERY_CLAIMS_USERINFO = "dl_claims_ui";

    /** Claims/Attributes having consent. */
    public static final String KEY_CONSENTED_CLAIMS = "cnsntd_claims";
    
    /** Whether consent has been enabled. */
    public static final String KEY_CONSENT_ENABLED = "cnsnt";

    /** Code Challenge. */
    public static final String KEY_CODE_CHALLENGE = "cc";

    /** Claims set for the claim. */
    private JWTClaimsSet tokenClaimsSet;

    /** Class logger. */
    @Nonnull private Logger log = LoggerFactory.getLogger(TokenClaimsSet.class);

    /**
     * Constructor.
     */
    protected TokenClaimsSet() {

    }

    /**
     * Constructor for token claims set.
     * 
     * @param tokenType Token type. Must not be NULL.
     * @param tokenID identifier for the token. Must not be NULL.
     * @param clientID Client Id of the rp. Must not be NULL.
     * @param issuer OP issuer value. Must not be NULL.
     * @param userPrincipal User Principal of the authenticated user. Must not be NULL.
     * @param subject subject of the authenticated user. Must not be NULL.
     * @param acr Authentication context class reference value of the authentication. May be NULL.
     * @param iat Issue time of the token. Must not be NULL.
     * @param exp Expiration time of the token. Must not be NULL.
     * @param nonce Nonce of the authentication request. May be NULL.
     * @param authTime Authentication time of the user. Must not be NULL.
     * @param redirectURI Validated redirect URI of the authentication request. Must not be NULL.
     * @param scope Scope of the authentication request. Must not be NULL.
     * @param claims Claims request of the authentication request. May be NULL.
     * @param dlClaims token delivery claims delivered both for id token and userinfo response. May be NULL.
     * @param dlClaimsID token delivery claims delivered for id token. May be NULL.
     * @param dlClaimsUI token delivery claims delivered for userinfo response. May be NULL.
     * @param consentedClaims consented claims. May be NULL.
     * @param codeChallenge Code Challenge. May be NULL.
     * @param consentEnabled Whether consent has been enabled.
     * @throws RuntimeException if called with not allowed null parameters
     */
    // Checkstyle: CyclomaticComplexity OFF
    // Checkstyle: ParameterNumber OFF
    protected TokenClaimsSet(@Nonnull final String tokenType, @Nonnull final String tokenID,
            @Nonnull final ClientID clientID, @Nonnull final String issuer, @Nonnull final String userPrincipal,
            @Nonnull final String subject, @Nullable final ACR acr, @Nonnull final Instant iat,
            @Nonnull final Instant exp, @Nullable final Nonce nonce, @Nonnull final Instant authTime,
            @Nonnull final URI redirectURI, @Nonnull final Scope scope, @Nullable final OIDCClaimsRequest claims,
            @Nullable final ClaimsSet dlClaims, @Nullable final ClaimsSet dlClaimsID,
            @Nullable final ClaimsSet dlClaimsUI, @Nullable final List<Object> consentedClaims,
            @Nullable final String codeChallenge, final boolean consentEnabled) {
        if (tokenType == null || tokenID == null || clientID == null || issuer == null || userPrincipal == null
                || iat == null || exp == null || authTime == null || redirectURI == null || scope == null
                || subject == null) {
            throw new RuntimeException("Invalid parameters, programming error");
        }
        tokenClaimsSet = new JWTClaimsSet.Builder().claim(KEY_TYPE, tokenType).jwtID(tokenID)
                .claim(KEY_CLIENTID, clientID.getValue()).issuer(issuer).subject(subject)
                .claim(KEY_USER_PRINCIPAL, userPrincipal).claim(KEY_ACR, acr == null ? null : acr.getValue())
                .issueTime(Date.from(iat)).expirationTime(Date.from(exp))
                .claim(KEY_NONCE, nonce == null ? null : nonce.getValue())
                .claim(KEY_AUTH_TIME, Date.from(authTime)).claim(KEY_REDIRECT_URI, redirectURI.toString())
                .claim(KEY_SCOPE, scope.toString()).claim(KEY_CLAIMS, claims == null ? null : claims.toJSONObject())
                .claim(KEY_DELIVERY_CLAIMS, dlClaims == null ? null : dlClaims.toJSONObject())
                .claim(KEY_DELIVERY_CLAIMS_IDTOKEN, dlClaimsID == null ? null : dlClaimsID.toJSONObject())
                .claim(KEY_DELIVERY_CLAIMS_USERINFO, dlClaimsUI == null ? null : dlClaimsUI.toJSONObject())
                .claim(KEY_CONSENTED_CLAIMS, consentedClaims).claim(KEY_CODE_CHALLENGE, codeChallenge)
                .claim(KEY_CONSENT_ENABLED, consentEnabled).build();

    }

    // Checkstyle: CyclomaticComplexity ON
    // Checkstyle: ParameterNumber ON

    /**
     * Helper to verify parsed claims are what is expected.
     * 
     * @param tokenType The type of the expected token. Must not be NULL.
     * @param tokenClaimsSet token claims set Must not be NULL.
     * @throws ParseException if claims set is not expected one.
     */
    // Checkstyle: CyclomaticComplexity OFF
    protected static void verifyParsedClaims(@Nonnull final String tokenType,
            @Nonnull final JWTClaimsSet tokenClaimsSet) throws ParseException {
        // Check existence and type of mandatory fields and values
        if (!tokenType.equals(tokenClaimsSet.getClaims().get(KEY_TYPE))) {
            throw new ParseException("claim type value not matching", 0);
        }
        // Mandatory fields
        if (tokenClaimsSet.getStringClaim(KEY_ISSUER) == null) {
            throw new ParseException("claim iss must exist and not be null", 0);
        }
        if (tokenClaimsSet.getStringClaim(KEY_USER_PRINCIPAL) == null) {
            throw new ParseException("claim prncpl must exist and not be null", 0);
        }
        if (tokenClaimsSet.getStringClaim(KEY_SUBJECT) == null) {
            throw new ParseException("claim sub must exist and not be null", 0);
        }
        if (tokenClaimsSet.getStringClaim(KEY_CLIENTID) == null) {
            throw new ParseException("claim clid must exist and not be null", 0);
        }
        if (tokenClaimsSet.getDateClaim(KEY_EXPIRATION_TIME) == null) {
            throw new ParseException("claim exp must exist and not be null", 0);
        }
        if (tokenClaimsSet.getDateClaim(KEY_ISSUED_AT) == null) {
            throw new ParseException("claim iat must exist and not be null", 0);
        }
        if (tokenClaimsSet.getStringClaim(KEY_AC_ID) == null) {
            throw new ParseException("claim jti must exist and not be null", 0);
        }
        if (tokenClaimsSet.getDateClaim(KEY_AUTH_TIME) == null) {
            throw new ParseException("claim auth_time must exist and not be null", 0);
        }
        if (tokenClaimsSet.getStringClaim(KEY_REDIRECT_URI) == null) {
            throw new ParseException("claim redirect_uri must exist and not be null", 0);
        }
        if (tokenClaimsSet.getStringClaim(KEY_SCOPE) == null) {
            throw new ParseException("claim scope must exist and not be null", 0);
        }
        if (tokenClaimsSet.getClaims().containsKey(KEY_ACR)) {
            tokenClaimsSet.getStringClaim(KEY_ACR);
        }
        if (tokenClaimsSet.getClaims().containsKey(KEY_CONSENTED_CLAIMS)
                && !(tokenClaimsSet.getClaim(KEY_CONSENTED_CLAIMS) instanceof List)) {
            throw new ParseException("consented claims is of wrong type", 0);
        }
        if (tokenClaimsSet.getClaims().containsKey(KEY_CONSENT_ENABLED)) {
            tokenClaimsSet.getBooleanClaim(KEY_CONSENT_ENABLED);
        }
        if (tokenClaimsSet.getClaims().containsKey(KEY_CLAIMS)) {
            tokenClaimsSet.getJSONObjectClaim(KEY_CLAIMS);
        }
        if (tokenClaimsSet.getClaims().containsKey(KEY_DELIVERY_CLAIMS)) {
            tokenClaimsSet.getJSONObjectClaim(KEY_DELIVERY_CLAIMS);
        }
        if (tokenClaimsSet.getClaims().containsKey(KEY_DELIVERY_CLAIMS_IDTOKEN)) {
            tokenClaimsSet.getJSONObjectClaim(KEY_DELIVERY_CLAIMS_IDTOKEN);
        }
        if (tokenClaimsSet.getClaims().containsKey(KEY_DELIVERY_CLAIMS_USERINFO)) {
            tokenClaimsSet.getJSONObjectClaim(KEY_DELIVERY_CLAIMS_USERINFO);
        }
        if (tokenClaimsSet.getClaims().containsKey(KEY_NONCE)) {
            tokenClaimsSet.getStringClaim(KEY_NONCE);
        }
        if (tokenClaimsSet.getClaims().containsKey(KEY_CODE_CHALLENGE)) {
            tokenClaimsSet.getStringClaim(KEY_CODE_CHALLENGE);
        }

    }
    // Checkstyle: CyclomaticComplexity ON

    /**
     * Serialize the token as JSON String.
     * 
     * @return token as JSON String
     */
    public String serialize() {
        return JSONObjectUtils.toJSONObject(tokenClaimsSet).toJSONString();
    }

    /**
     * Serialize the token as JSON String wrapped with sealer.
     * 
     * @param dataSealer data sealer to wrap the JSON serialization
     * @return token as JSON String wrapped with sealer
     * @throws DataSealerException is thrown if unwrapping fails
     */
    public String serialize(@Nonnull final DataSealer dataSealer) throws DataSealerException {
        return dataSealer.wrap(serialize(), Instant.ofEpochMilli(tokenClaimsSet.getExpirationTime().getTime()));
    }
    
    /**
     * Set the token claims set.
     * 
     * @param claimsSet What to set.
     */
    public void setClaimsSet(final @Nonnull JWTClaimsSet claimsSet) {
        tokenClaimsSet = claimsSet;
    }

    /**
     * Get the token claims set.
     * 
     * @return token claims set
     */
    @Nonnull
    public JWTClaimsSet getClaimsSet() {
        return tokenClaimsSet;
    }

    /**
     * Check if the token is expired.
     * 
     * @return true if the token is expired, otherwise false.
     */
    public boolean isExpired() {
        return tokenClaimsSet.getExpirationTime().before(new Date());
    }

    /**
     * Get expiration time of the token.
     * 
     * @return expiration time of the token.
     */
    @Nonnull
    public Instant getExp() {
        return tokenClaimsSet.getExpirationTime().toInstant();
    }

    /**
     * Get redirect uri of the request.
     * 
     * @return redirect uri of the request, null if not located.
     */
    @Nonnull
    public URI getRedirectURI() {
        try {
            return URI.create(tokenClaimsSet.getStringClaim(KEY_REDIRECT_URI));
        } catch (final ParseException e) {
            log.error("error parsing redirect uri from token", e.getMessage());
        }
        // should never happen, programming error.
        return null;
    }

    /**
     * Get acr of the performed authentication.
     * 
     * @return acr of the performed authentication.
     */
    @Nonnull
    public String getACR() {
        return (String) tokenClaimsSet.getClaim(KEY_ACR);
    }
    
    /**
     * Get type of the claims set.
     * 
     * @return Type of the claims set.
     */
    @Nonnull
    public String getType() {
        return (String) tokenClaimsSet.getClaim(KEY_TYPE);
    }

    /**
     * Get principal of the user.
     * 
     * @return principal of the user.
     */
    @Nonnull
    public String getPrincipal() {
        return (String) tokenClaimsSet.getClaim(KEY_USER_PRINCIPAL);
    }

    /**
     * Get auth time of the user.
     * 
     * @return auth time of the user.
     */
    @Nonnull
    public Instant getAuthenticationTime() {
        try {
            return tokenClaimsSet.getDateClaim(KEY_AUTH_TIME).toInstant();
        } catch (final ParseException e) {
            log.error("Error parsing auth time {}", tokenClaimsSet.getClaim(KEY_AUTH_TIME));
            // should never happen, programming error.
            return null;
        }
    }

    /**
     * Get nonce of the authentication request.
     * 
     * @return nonce of the authentication request.
     */
    @Nonnull
    public Nonce getNonce() {
        if (tokenClaimsSet.getClaim(KEY_NONCE) == null) {
            return null;
        }
        return new Nonce((String) tokenClaimsSet.getClaim(KEY_NONCE));
    }

    /**
     * Get claims request of the authentication request.
     * 
     * @return claims request in authentication request, null if not existing.
     */
    @Nullable
    public OIDCClaimsRequest getClaimsRequest() {
        if (tokenClaimsSet.getClaim(KEY_CLAIMS) == null) {
            return null;
        }
        try {
            return OIDCClaimsRequest.parse(new JSONObject(tokenClaimsSet.getJSONObjectClaim(KEY_CLAIMS)));
        } catch (final ParseException | com.nimbusds.oauth2.sdk.ParseException e) {
            log.error("Error parsing claims request {}", tokenClaimsSet.getClaim(KEY_CLAIMS));
            return null;
        }
    }

    /**
     * Get token delivery claims.
     * 
     * @return token delivery claims
     */
    public ClaimsSet getDeliveryClaims() {
        final TokenDeliveryClaimsClaimsSet claimsSet = new TokenDeliveryClaimsClaimsSet();
        try {
            final Map<String, Object> claims = tokenClaimsSet.getJSONObjectClaim(KEY_DELIVERY_CLAIMS);
            if (claims == null) {
                return null;
            }
            claimsSet.putAll(claims);
        } catch (final ParseException e) {
            log.error("Error parsing delivery claims {}", tokenClaimsSet.getClaim(KEY_DELIVERY_CLAIMS));
            return null;
        }
        return claimsSet;
    }

    /**
     * Get id token token delivery claims.
     * 
     * @return id token token delivery claims
     */
    public ClaimsSet getIDTokenDeliveryClaims() {
        final TokenDeliveryClaimsClaimsSet claimsSet = new TokenDeliveryClaimsClaimsSet();
        try {
            final Map<String, Object> claims = tokenClaimsSet.getJSONObjectClaim(KEY_DELIVERY_CLAIMS_IDTOKEN);
            if (claims == null) {
                return null;
            }
            claimsSet.putAll(claims);
        } catch (final ParseException e) {
            log.error("Error parsing id token delivery claims {}",
                    tokenClaimsSet.getClaim(KEY_DELIVERY_CLAIMS_IDTOKEN));
            return null;
        }
        return claimsSet;
    }

    /**
     * Get user info response token delivery claims.
     * 
     * @return user info response token delivery claims
     */
    public ClaimsSet getUserinfoDeliveryClaims() {
        final TokenDeliveryClaimsClaimsSet claimsSet = new TokenDeliveryClaimsClaimsSet();
        try {
            final Map<String, Object> claims = tokenClaimsSet.getJSONObjectClaim(KEY_DELIVERY_CLAIMS_USERINFO);
            if (claims == null) {
                return null;
            }
            claimsSet.putAll(claims);
        } catch (final ParseException e) {
            log.error("Error parsing id token delivery claims {}",
                    tokenClaimsSet.getClaim(KEY_DELIVERY_CLAIMS_USERINFO));
            return null;
        }
        return claimsSet;
    }

    /**
     * Get consented claims.
     * 
     * @return consented claims
     */
    public List<Object> getConsentedClaims() {
        return (List<Object>) tokenClaimsSet.getClaim(KEY_CONSENTED_CLAIMS);
    }

    /**
     * Get whether consent has been enabled.
     * 
     * @return whether consent has been enabled
     */
    public boolean isConsentEnabled() {
        try {
            return tokenClaimsSet.getBooleanClaim(KEY_CONSENT_ENABLED).booleanValue();
        } catch (final ParseException e) {
            log.error("Error parsing scope in request {}", tokenClaimsSet.getClaim(KEY_CONSENT_ENABLED));
            // should never happen, programming error.
            return false;
        }
    }

    /**
     * Get scope of the authentication request.
     * 
     * @return scope of the authentication request.
     */
    @Nonnull
    public Scope getScope() {
        try {
            return Scope.parse(tokenClaimsSet.getStringClaim(KEY_SCOPE));
        } catch (final ParseException e) {
            log.error("Error parsing scope in request {}", tokenClaimsSet.getClaim(KEY_SCOPE));
            // should never happen, programming error.
            return null;
        }
    }

    /**
     * Get code challenge of the authentication request.
     * 
     * @return code challenge of the authentication request.
     */
    @Nonnull
    public String getCodeChallenge() {
        if (tokenClaimsSet.getClaim(KEY_CODE_CHALLENGE) == null) {
            return null;
        }
        return (String) tokenClaimsSet.getClaim(KEY_CODE_CHALLENGE);
    }

    /**
     * Get the id of the token.
     * 
     * @return id of the token
     */
    @Nonnull
    public String getID() {
        return tokenClaimsSet.getJWTID();
    }

    /**
     * Get Client ID of the token.
     * 
     * @return Client ID of the token
     */
    @Nonnull
    public ClientID getClientID() {
        return new ClientID((String) tokenClaimsSet.getClaim(KEY_CLIENTID));
    }

    /**
     * Abstract builder to extend builders from that are instantiating claims sets extending TokenClaimsSet.
     * 
     * @param <T> claim set type
     */
    public abstract static class Builder<T extends TokenClaimsSet> {

        // Checkstyle: VisibilityModifier OFF

        /** Generator for pseudo unique identifier for the claims set. */
        @Nonnull
        protected IdentifierGenerationStrategy idGen;

        /** Client Id of the rp. */
        @Nonnull
        protected ClientID rpId;

        /** OP issuer value. */
        @Nonnull
        protected String iss;

        /** User Principal of the authenticated user. */
        @Nonnull
        protected String usrPrincipal;

        /** Subject claim value of the authenticated user. */
        @Nonnull
        protected String sub;

        /** Authentication context class reference value of the authentication. */
        @Nonnull
        protected ACR acr;

        /** Issue time of the claims set. */
        @Nonnull
        protected Instant iat;

        /** Expiration time of the claims set. */
        @Nonnull
        protected Instant exp;

        /** Authentication time of the user. */
        @Nonnull
        protected Instant authTime;

        /** Validated redirect URI of the authentication request. */
        @Nonnull
        protected URI redirect;

        /** Scope of the authentication request. */
        @Nonnull
        protected Scope reqScope;

        /** Nonce of the authentication request. */
        @Nullable
        protected Nonce nonce;

        /** Claims request of the authentication request. */
        @Nullable
        protected OIDCClaimsRequest claims;

        /** Token delivery claims delivered both for id token and userinfo response. */
        @Nullable
        protected ClaimsSet dlClaims;

        /** Token delivery claims delivered for id token. */
        @Nullable
        protected ClaimsSet dlClaimsID;

        /** Token delivery claims delivered for userinfo response. */
        @Nullable
        protected ClaimsSet dlClaimsUI;

        /** consented claims. */
        @Nullable
        protected List<Object> cnsntdClaims;
        
        /** Has consent been asked from the end-user. */
        protected boolean cnsntEnabled;

        /** Code challenge. */
        @Nullable
        protected String codeChallenge;
        
        // Checkstyle: VisibilityModifier ON

        /**
         * Constructor for authorize code builder.
         * 
         * @param idGenerator Generator for pseudo unique identifier for the claims set. Must not be NULL.
         * @param clientID Client Id of the rp. Must not be NULL.
         * @param issuer OP issuer value. Must not be NULL.
         * @param userPrincipal User Principal of the authenticated user. Must not be NULL.
         * @param subject subject of the authenticated user. Must not be NULL
         * @param issuedAt Issue time of the authorize code. Must not be NULL.
         * @param expiresAt Expiration time of the authorize code. Must not be NULL.
         * @param authenticationTime Authentication time of the user. Must not be NULL.
         * @param redirectURI Validated redirect URI of the authentication request. Must not be NULL.
         * @param scope Scope of the authentication request. Must not be NULL.
         */
        // Checkstyle: ParameterNumber OFF
        protected Builder(@Nonnull final IdentifierGenerationStrategy idGenerator, @Nonnull final ClientID clientID,
                @Nonnull final String issuer, @Nonnull final String userPrincipal, @Nonnull final String subject,
                @Nonnull final Instant issuedAt, @Nonnull final Instant expiresAt,
                @Nonnull final Instant authenticationTime, @Nonnull final URI redirectURI, @Nonnull final Scope scope) {

            idGen = idGenerator;
            rpId = clientID;
            iss = issuer;
            usrPrincipal = userPrincipal;
            sub = subject;
            iat = issuedAt;
            exp = expiresAt;
            authTime = authenticationTime;
            redirect = redirectURI;
            reqScope = scope;

        }
        // Checkstyle: ParameterNumber ON


        /**
         * Set authentication context class reference value of the authentication.
         * 
         * @param authenticationContextReference authentication context class reference value of the authentication.
         * 
         * @return the builder
         */
        public Builder<T> setACR(@Nullable final ACR authenticationContextReference) {
            acr = authenticationContextReference;
            return this;
        }

        /**
         * Set nonce of the authentication request.
         * 
         * @param requestNonce nonce of the authentication request.
         * 
         * @return the builder
         */
        public Builder<T> setNonce(@Nullable final Nonce requestNonce) {
            nonce = requestNonce;
            return this;
        }

        /**
         * Set claims request of the authentication request.
         * 
         * @param requestedClaims claims request of the authentication request.
         * 
         * @return the builder
         */
        public Builder<T> setClaims(@Nullable final OIDCClaimsRequest requestedClaims) {
            claims = requestedClaims;
            return this;
        }

        /**
         * Set token delivery claims delivered both for id token and userinfo response.
         * 
         * @param deliveryClaims token delivery claims delivered both for id token and userinfo response.
         * 
         * @return the builder
         */
        public Builder<T> setDlClaims(@Nullable final ClaimsSet deliveryClaims) {
            dlClaims = deliveryClaims;
            return this;
        }

        /**
         * Set token delivery claims delivered for id token.
         * 
         * @param deliveryClaimsIDToken token delivery claims delivered for id token
         * 
         * @return the builder
         */
        public Builder<T> setDlClaimsID(@Nullable final ClaimsSet deliveryClaimsIDToken) {
            dlClaimsID = deliveryClaimsIDToken;
            return this;
        }

        /**
         * Set token delivery claims delivered for userinfo response.
         * 
         * @param deliveryClaimsUserInfo token delivery claims delivered for userinfo response
         * 
         * @return the builder
         */
        public Builder<T> setDlClaimsUI(@Nullable final ClaimsSet deliveryClaimsUserInfo) {
            dlClaimsUI = deliveryClaimsUserInfo;
            return this;
        }

        /**
         * Set consented claims.
         * 
         * @param consentedClaims consented claims
         * 
         * @return the builder
         */
        public Builder<T> setConsentedClaims(@Nullable final List<Object> consentedClaims) {
            cnsntdClaims = consentedClaims;
            return this;
        }

        /**
         * Set whether consent has been enabled.
         * 
         * @param consentEnabled whether consent has been enabled.
         * 
         * @return the builder
         */
        public Builder<T> setConsentEnabled(final boolean consentEnabled) {
            cnsntEnabled = consentEnabled;
            return this;
        }

        /**
         * Set code challenge.
         * 
         * @param challenge code challenge
         * 
         * @return the builder
         */
        public Builder<T> setCodeChallenge(@Nullable final String challenge) {
            codeChallenge = challenge;
            return this;
        }

        /**
         * Builds claims set.
         * 
         * @return claims set instance.
         */
        public abstract T build();
    }

}