/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.shared.service.security.impl;

import javax.annotation.Nonnull;

import net.shibboleth.shared.annotation.ParameterName;
import net.shibboleth.shared.component.ComponentInitializationException;
import net.shibboleth.shared.logic.Constraint;
import net.shibboleth.shared.security.AccessControl;
import net.shibboleth.shared.security.AccessControlService;
import net.shibboleth.shared.spring.service.AbstractServiceableComponent;

/**
 * This class wraps an {@link AccessControlService} in a
 * {@link net.shibboleth.shared.service.ServiceableComponent}.
 */
public class ReloadingAccessControlService extends AbstractServiceableComponent<AccessControlService>
        implements AccessControlService {

    /** The embedded service. */
    @Nonnull private final AccessControlService service;

    /**
     * Constructor.
     * 
     * @param svc the embedded service
     */
    public ReloadingAccessControlService(@Nonnull @ParameterName(name="svc") final AccessControlService svc) {
        service = Constraint.isNotNull(svc, "AccessControlService cannot be null");
    }

    /** {@inheritDoc} */
    @Override
    protected void doInitialize() throws ComponentInitializationException {
        final String id = service.getId();
        if (id != null) {
            setId(id);
        } else if (getId() == null){
            setId("UnNamedReloadingAccessControlService");
        }
        super.doInitialize();
    }

    /** {@inheritDoc} */
    @Override
    @Nonnull public AccessControl getInstance(@Nonnull final String name) {
        return service.getInstance(name);
    }

    /** {@inheritDoc} */
    @Override
    @Nonnull public AccessControlService getComponent() {
        return this;
    }

}