/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.shared.spring.service;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.springframework.context.ApplicationContext;

import net.shibboleth.shared.component.ComponentInitializationException;
import net.shibboleth.shared.logic.NonnullFunction;
import net.shibboleth.shared.service.ServiceException;

/**
 * Strategy for summoning up an {@link ApplicationContextServiceableComponent} wrapper
 * around a populated {@link ApplicationContext}.
 * 
 * @since 5.4.0
 */
public class ApplicationContextServiceStrategy implements
        NonnullFunction<ApplicationContext, AbstractServiceableComponent<ApplicationContext>> {
    
    /** {@inheritDoc} */
    @Nonnull
    public AbstractServiceableComponent<ApplicationContext> apply(@Nullable final ApplicationContext appContext) {

        if (appContext != null) {
            final ApplicationContextServiceableComponent wrapper = new ApplicationContextServiceableComponent();
            wrapper.setApplicationContext(appContext);
            final String id = appContext.getId();
            if (id != null) {
                wrapper.setId(id);
            } else {
                wrapper.setId(appContext.getApplicationName());
            }
            try {
                wrapper.initialize();
                return wrapper;
            } catch (final ComponentInitializationException e) {
                throw new ServiceException("Unable to initialize component wraper for ApplicationContext "
                        + appContext.getDisplayName(), e);
            }
        } else {
            throw new ServiceException("Unable to initialize component wraper for absent ApplicationContext");
        }
    }
    
}