/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.shared.spring.service;

import java.util.Collection;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.springframework.context.ApplicationContext;

import net.shibboleth.shared.annotation.ParameterName;
import net.shibboleth.shared.component.ComponentInitializationException;
import net.shibboleth.shared.logic.Constraint;
import net.shibboleth.shared.logic.NonnullFunction;
import net.shibboleth.shared.service.ServiceException;
import net.shibboleth.shared.spring.service.impl.SpringServiceableComponent;

/**
 * Strategy to create {@link AbstractServiceableComponent}s from the {@link ApplicationContext}.
 * 
 * @param <T> the service type to look for
 */
public class ClassBasedServiceStrategy<T>
        implements NonnullFunction<ApplicationContext, AbstractServiceableComponent<T>> {

    /** The class we are looking for. */
    @Nonnull private final Class<T> serviceClaz;

    /**
     * Constructor.
     * 
     * @param serviceableClaz what to look for.
     */
    public ClassBasedServiceStrategy(
            @ParameterName(name="serviceableClaz") @Nonnull final Class<T> serviceableClaz) {
        serviceClaz = Constraint.isNotNull(serviceableClaz, "Serviceable Class cannot be null");
    }

    /** {@inheritDoc} */
    @Nonnull public AbstractServiceableComponent<T> apply(@Nullable final ApplicationContext appContext) {
        
        if (appContext == null) {
            throw new ServiceException("Input ApplicationContext was null");
        }
        
        final Collection<T> components = appContext.getBeansOfType(serviceClaz).values();

        if (components.size() == 0) {
            throw new ServiceException("Reload did not produce any bean of type " + serviceClaz.getName());
        }
        if (components.size() > 1) {
            throw new ServiceException("Reload produced " + components.size() + " ServiceableComponents");
        }
        final T value = components.iterator().next();
        assert value != null;
        final SpringServiceableComponent<T> result = new SpringServiceableComponent<>(value);
        result.setApplicationContext(appContext);
        try {
            result.initialize();
        } catch (final ComponentInitializationException e) {
            throw new ServiceException("Unable to initialize service type " + serviceClaz.getName() + " from "
                    + appContext.getDisplayName(), e);
        }
        return result;
    }
    
}