/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.shared.spring.service.impl;

import javax.annotation.Nonnull;

import net.shibboleth.shared.component.IdentifiedComponent;
import net.shibboleth.shared.primitive.LoggerFactory;
import net.shibboleth.shared.service.ServiceableComponent;
import net.shibboleth.shared.spring.service.AbstractServiceableComponent;

/**
 * A container @{link {@link ServiceableComponent} which is (usually) created during spring service
 * creation as part of service (re)load.
 * 
 * @param <T> component type
 */
public class SpringServiceableComponent<T> extends AbstractServiceableComponent<T> {

    /** The component we are encapsulating. */
    @Nonnull private final T theComponent;
    
    /**
     * Constructor.
     *
     * @param what the object we are wrapping.
     */
    public SpringServiceableComponent(@Nonnull final T what) {
        theComponent = what;
        String id = null;
        if (what instanceof IdentifiedComponent c) {
            id = c.getId();
        }
    	if (id != null) {
            setId(id);
        }
    	else {
    		final String generatedId = what.toString() + Long.toString(System.currentTimeMillis()); 
    		LoggerFactory.getLogger(SpringServiceableComponent.class).debug("Generated id {} for object of type {}", what.getClass(), generatedId);
    		setId(generatedId);
    	}
    }
    
    /** {@inheritDoc} */
    @Override
    @Nonnull public T getComponent() {
        return theComponent;
    }

}