/*
 * Decompiled with CFR 0.152.
 */
package org.hibernate.search.store;

import java.io.File;
import java.io.IOException;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.util.Properties;
import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.analysis.SimpleAnalyzer;
import org.apache.lucene.index.IndexReader;
import org.apache.lucene.index.IndexWriter;
import org.apache.lucene.store.Directory;
import org.apache.lucene.store.FSDirectory;
import org.apache.lucene.store.LockFactory;
import org.apache.lucene.store.MMapDirectory;
import org.apache.lucene.store.NIOFSDirectory;
import org.apache.lucene.store.NativeFSLockFactory;
import org.apache.lucene.store.NoLockFactory;
import org.apache.lucene.store.SimpleFSDirectory;
import org.apache.lucene.store.SimpleFSLockFactory;
import org.apache.lucene.store.SingleInstanceLockFactory;
import org.hibernate.annotations.common.util.StringHelper;
import org.hibernate.search.SearchException;
import org.hibernate.search.store.LockFactoryFactory;
import org.hibernate.search.util.ClassLoaderHelper;
import org.hibernate.search.util.LoggerFactory;
import org.slf4j.Logger;

public final class DirectoryProviderHelper {
    private static final Logger log = LoggerFactory.make();
    private static final String ROOT_INDEX_PROP_NAME = "sourceBase";
    private static final String RELATIVE_INDEX_PROP_NAME = "source";
    private static final String COPY_BUFFER_SIZE_PROP_NAME = "buffer_size_on_copy";
    private static final String LOCKING_STRATEGY_PROP_NAME = "locking_strategy";
    private static final String FS_DIRECTORY_TYPE_PROP_NAME = "filesystem_access_type";
    private static final String INDEX_BASE_PROP_NAME = "indexBase";
    private static final String INDEX_NAME_PROP_NAME = "indexName";
    private static final String REFRESH_PROP_NAME = "refresh";

    private DirectoryProviderHelper() {
    }

    public static File getSourceDirectory(String directoryProviderName, Properties properties, boolean needWritePermissions) {
        File sourceDirectory;
        String root = properties.getProperty(ROOT_INDEX_PROP_NAME);
        String relative = properties.getProperty(RELATIVE_INDEX_PROP_NAME);
        if (log.isTraceEnabled()) {
            log.trace("Guess source directory from {} {} and {} {}", new Object[]{ROOT_INDEX_PROP_NAME, root != null ? root : "<null>", RELATIVE_INDEX_PROP_NAME, relative != null ? relative : "<null>"});
        }
        if (relative == null) {
            relative = directoryProviderName;
        }
        if (StringHelper.isEmpty((String)root)) {
            log.debug("No root directory, go with relative " + relative);
            sourceDirectory = new File(relative);
            if (!sourceDirectory.isDirectory()) {
                throw new SearchException("Unable to read source directory: " + relative);
            }
        } else {
            File rootDir = new File(root);
            DirectoryProviderHelper.makeSanityCheckedDirectory(rootDir, directoryProviderName, needWritePermissions);
            sourceDirectory = new File(root, relative);
            DirectoryProviderHelper.makeSanityCheckedDirectory(sourceDirectory, directoryProviderName, needWritePermissions);
            log.debug("Got directory from root + relative");
        }
        return sourceDirectory;
    }

    public static FSDirectory createFSIndex(File indexDir, Properties properties) throws IOException {
        LockFactory lockFactory = DirectoryProviderHelper.createLockFactory(indexDir, properties);
        FSDirectoryType fsDirectoryType = FSDirectoryType.getType(properties);
        FSDirectory fsDirectory = fsDirectoryType.getDirectory(indexDir, null);
        fsDirectory.setLockFactory(lockFactory);
        if (!IndexReader.indexExists((Directory)fsDirectory)) {
            log.debug("Initialize index: '{}'", (Object)indexDir.getAbsolutePath());
            IndexWriter.MaxFieldLength fieldLength = new IndexWriter.MaxFieldLength(10000);
            IndexWriter iw = new IndexWriter((Directory)fsDirectory, (Analyzer)new SimpleAnalyzer(), true, fieldLength);
            iw.close();
        }
        return fsDirectory;
    }

    public static LockFactory createLockFactory(File indexDir, Properties dirConfiguration) {
        String defaultStrategy = indexDir == null ? "single" : "simple";
        String lockFactoryName = dirConfiguration.getProperty(LOCKING_STRATEGY_PROP_NAME, defaultStrategy);
        if ("simple".equals(lockFactoryName)) {
            if (indexDir == null) {
                throw new SearchException("To use \"simple\" as a LockFactory strategy an indexBase path must be set");
            }
            try {
                return new SimpleFSLockFactory(indexDir);
            }
            catch (IOException e) {
                throw new SearchException("Could not initialize SimpleFSLockFactory", e);
            }
        }
        if ("native".equals(lockFactoryName)) {
            if (indexDir == null) {
                throw new SearchException("To use \"native\" as a LockFactory strategy an indexBase path must be set");
            }
            try {
                return new NativeFSLockFactory(indexDir);
            }
            catch (IOException e) {
                throw new SearchException("Could not initialize NativeFSLockFactory", e);
            }
        }
        if ("single".equals(lockFactoryName)) {
            return new SingleInstanceLockFactory();
        }
        if ("none".equals(lockFactoryName)) {
            return new NoLockFactory();
        }
        LockFactoryFactory lockFactoryFactory = ClassLoaderHelper.instanceFromName(LockFactoryFactory.class, lockFactoryName, DirectoryProviderHelper.class, LOCKING_STRATEGY_PROP_NAME);
        return lockFactoryFactory.createLockFactory(indexDir, dirConfiguration);
    }

    public static File getVerifiedIndexDir(String annotatedIndexName, Properties properties, boolean verifyIsWritable) {
        String indexBase = properties.getProperty(INDEX_BASE_PROP_NAME, ".");
        String indexName = properties.getProperty(INDEX_NAME_PROP_NAME, annotatedIndexName);
        File baseIndexDir = new File(indexBase);
        DirectoryProviderHelper.makeSanityCheckedDirectory(baseIndexDir, indexName, verifyIsWritable);
        File indexDir = new File(baseIndexDir, indexName);
        DirectoryProviderHelper.makeSanityCheckedDirectory(indexDir, indexName, verifyIsWritable);
        return indexDir;
    }

    private static void makeSanityCheckedDirectory(File directory, String indexName, boolean verifyIsWritable) {
        if (!directory.exists()) {
            log.warn("Index directory not found, creating: '" + directory.getAbsolutePath() + "'");
            if (!directory.mkdirs()) {
                throw new SearchException("Unable to create index directory: " + directory.getAbsolutePath() + " for index " + indexName);
            }
        } else if (!directory.isDirectory()) {
            throw new SearchException("Unable to initialize index: " + indexName + ": " + directory.getAbsolutePath() + " is a file.");
        }
        if (verifyIsWritable && !directory.canWrite()) {
            throw new SearchException("Cannot write into index directory: " + directory.getAbsolutePath() + " for index " + indexName);
        }
    }

    static long getRefreshPeriod(Properties properties, String directoryProviderName) {
        long period;
        String refreshPeriod = properties.getProperty(REFRESH_PROP_NAME, "3600");
        try {
            period = Long.parseLong(refreshPeriod);
        }
        catch (NumberFormatException nfe) {
            throw new SearchException("Unable to initialize index: " + directoryProviderName + "; refresh period is not numeric.", nfe);
        }
        log.debug("Refresh period: {} seconds", (Object)period);
        return period * 1000L;
    }

    public static long getCopyBufferSize(String directoryProviderName, Properties properties) {
        String value = properties.getProperty(COPY_BUFFER_SIZE_PROP_NAME);
        long size = 0x1000000L;
        if (value != null) {
            try {
                size = Long.parseLong(value) * 1024L * 1024L;
            }
            catch (NumberFormatException nfe) {
                throw new SearchException("Unable to initialize index " + directoryProviderName + "; " + COPY_BUFFER_SIZE_PROP_NAME + " is not numeric.", nfe);
            }
            if (size <= 0L) {
                throw new SearchException("Unable to initialize index " + directoryProviderName + "; " + COPY_BUFFER_SIZE_PROP_NAME + " needs to be greater than zero.");
            }
        }
        return size;
    }

    /*
     * This class specifies class file version 49.0 but uses Java 6 signatures.  Assumed Java 6.
     */
    private static enum FSDirectoryType {
        AUTO(null),
        SIMPLE(SimpleFSDirectory.class),
        NIO(NIOFSDirectory.class),
        MMAP(MMapDirectory.class);

        private Class<?> fsDirectoryClass;

        private FSDirectoryType(Class<?> fsDirectoryClass) {
            this.fsDirectoryClass = fsDirectoryClass;
        }

        public FSDirectory getDirectory(File indexDir, LockFactory factory) throws IOException {
            FSDirectory directory;
            if (this.fsDirectoryClass == null) {
                directory = FSDirectory.open((File)indexDir, (LockFactory)factory);
            } else {
                try {
                    Constructor<?> constructor = this.fsDirectoryClass.getConstructor(File.class, LockFactory.class);
                    directory = (FSDirectory)constructor.newInstance(indexDir, factory);
                }
                catch (NoSuchMethodException e) {
                    throw new SearchException("Unable to find appropriate FSDirectory constructor", e);
                }
                catch (InstantiationException e) {
                    throw new SearchException("Unable to instantiate FSDirectory class " + this.fsDirectoryClass.getName(), e);
                }
                catch (IllegalAccessException e) {
                    throw new SearchException("Unable to instantiate FSDirectory class " + this.fsDirectoryClass.getName(), e);
                }
                catch (InvocationTargetException e) {
                    throw new SearchException("Unable to instantiate FSDirectory class " + this.fsDirectoryClass.getName(), e);
                }
            }
            return directory;
        }

        public static FSDirectoryType getType(Properties properties) {
            FSDirectoryType fsDirectoryType;
            String fsDirectoryTypeValue = properties.getProperty(DirectoryProviderHelper.FS_DIRECTORY_TYPE_PROP_NAME);
            if (StringHelper.isNotEmpty((String)fsDirectoryTypeValue)) {
                try {
                    fsDirectoryType = Enum.valueOf(FSDirectoryType.class, fsDirectoryTypeValue.toUpperCase());
                }
                catch (IllegalArgumentException e) {
                    throw new SearchException("Invalid option value for filesystem_access_type: " + fsDirectoryTypeValue);
                }
            } else {
                fsDirectoryType = AUTO;
            }
            return fsDirectoryType;
        }
    }
}

