/*
 * Hibernate Search, full-text search for your domain model
 *
 * License: GNU Lesser General Public License (LGPL), version 2.1 or later
 * See the lgpl.txt file in the root directory or <http://www.gnu.org/licenses/lgpl-2.1.html>.
 */
package org.hibernate.search.mapper.pojo.processing.impl;

import org.hibernate.search.engine.backend.document.DocumentElement;
import org.hibernate.search.engine.environment.bean.BeanHolder;
import org.hibernate.search.mapper.pojo.bridge.PropertyBridge;
import org.hibernate.search.mapper.pojo.processing.spi.PojoIndexingProcessorSessionContext;
import org.hibernate.search.util.common.impl.Closer;
import org.hibernate.search.util.common.impl.ToStringTreeBuilder;

/**
 * A node inside a {@link PojoIndexingProcessor} responsible for applying a {@link PropertyBridge}.
 *
 * @param <P> The type of the processed property.
 */
public class PojoIndexingProcessorPropertyBridgeNode<P> extends PojoIndexingProcessor<P> {

	private final BeanHolder<? extends PropertyBridge> bridgeHolder;

	public PojoIndexingProcessorPropertyBridgeNode(BeanHolder<? extends PropertyBridge> bridgeHolder) {
		this.bridgeHolder = bridgeHolder;
	}

	@Override
	public void close() {
		try ( Closer<RuntimeException> closer = new Closer<>() ) {
			closer.push( holder -> holder.get().close(), bridgeHolder );
			closer.push( BeanHolder::close, bridgeHolder );
		}
	}

	@Override
	public void appendTo(ToStringTreeBuilder builder) {
		builder.attribute( "operation", "apply property bridge" );
		builder.attribute( "bridge", bridgeHolder );
	}

	@Override
	public final void process(DocumentElement target, P source, PojoIndexingProcessorSessionContext sessionContext) {
		bridgeHolder.get().write( target, source, sessionContext.propertyBridgeWriteContext() );
	}

}
