/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors. 
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */ 
package org.jboss.aop.asintegration.jboss5;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.WeakHashMap;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;

import org.jboss.aop.Domain;
import org.jboss.classloader.spi.ClassLoaderDomain;
import org.jboss.classloading.spi.dependency.Module;
import org.jboss.classpool.plugins.jbosscl.VFSClassLoaderDomainRegistry;

/**
 * 
 * @author <a href="flavia.rainone@jboss.com">Flavia Rainone</a>
 * @version $Revision: 1.1 $
 */
public class AOPVFSClassLoaderDomainRegistry extends VFSClassLoaderDomainRegistry implements AOPDomainRegistry
{
   /** aopDomains by classloader */
   private Map<ClassLoader, ScopedVFSClassLoaderDomain> aopDomainsByClassLoader = new WeakHashMap<ClassLoader, ScopedVFSClassLoaderDomain>();
   
   /** aopDomains by classloader domain */
   private Map<ClassLoaderDomain, List<ScopedVFSClassLoaderDomain>> aopDomainsByClassLoaderDomain = new WeakHashMap<ClassLoaderDomain, List<ScopedVFSClassLoaderDomain>>();

   private Map<ClassLoaderDomain, ReadWriteLock> aopDomainsLocksByClassLoaderDomain = new WeakHashMap<ClassLoaderDomain, ReadWriteLock>();

   private ScopedVFSClassLoaderDomain initDomain = null;

   public List<ScopedVFSClassLoaderDomain> getAOPDomainsForClassLoaderDomain(ClassLoaderDomain domain)
   {
      List<ScopedVFSClassLoaderDomain> domains = aopDomainsByClassLoaderDomain.get(domain);
      if (domains != null)
      {
         return domains;
      }
      return Collections.emptyList();
   }

   public ReadWriteLock getAOPDomainsLockForClassLoaderDomain(ClassLoaderDomain domain)
   {
      return aopDomainsLocksByClassLoaderDomain.get(domain);
   }

   public synchronized Domain getRegisteredDomain(ClassLoader cl)
   {
      return aopDomainsByClassLoader.get(cl);
   }


   public synchronized void setDomain(ScopedVFSClassLoaderDomain domain)
   {
      this.initDomain = domain;
   }
   
   @Override
   protected void validateInitMaps(ClassLoader loader, ClassLoaderDomain loaderDomain, Module module, ClassLoader parentUnitLoader)
   {
      if (loaderDomain == null && initDomain != null)
      {
         throw new IllegalStateException("Have " + domain + " but no classloader domain");
      }
   }
   
   @Override
   protected void initMapsDone(ClassLoader loader, ClassLoaderDomain loaderDomain, Module module, ClassLoader parentUnitLoader)
   {
      if (initDomain != null)
      {
         aopDomainsByClassLoader.put(loader, initDomain);

         ReadWriteLock lock = aopDomainsLocksByClassLoaderDomain.get(loaderDomain);
         if (lock == null)
         {
            lock = new ReentrantReadWriteLock();
            aopDomainsLocksByClassLoaderDomain.put(loaderDomain, lock);
         }
         
         lock.writeLock().lock();
         try
         {
            List<ScopedVFSClassLoaderDomain> aopDomains = aopDomainsByClassLoaderDomain.get(loaderDomain);
            if (aopDomains == null)
            {
               synchronized (aopDomainsByClassLoaderDomain)
               {
                  aopDomains = new CopyOnWriteArrayList<ScopedVFSClassLoaderDomain>();
                  if (!aopDomains.contains(loaderDomain))
                  {
                     aopDomainsByClassLoaderDomain.put(loaderDomain, aopDomains);
                  }
               }
            }
            aopDomains.add(initDomain);
         }
         finally
         {
            lock.writeLock().unlock();
         }
      }
   }
   
   @Override
   public synchronized void cleanupLoaderDone(ClassLoader loader, ClassLoaderDomain loaderDomain, int domainReferences)
   {
      if (loaderDomain != null)
      {
         if (domainReferences == 0)
         {
            //aopDomainsByClassLoaderDomain.remove(loaderDomain);
         }
         ScopedVFSClassLoaderDomain aopDomain = aopDomainsByClassLoader.remove(loader);
         List<ScopedVFSClassLoaderDomain> aopDomains = aopDomainsByClassLoaderDomain.get(loaderDomain);
         if (aopDomains != null)
         {
            aopDomains.remove(aopDomain);
         }
         if (aopDomain != null)
         {
            aopDomain.getParent().unsubscribeSubDomain(aopDomain);
         }
      }
      aopDomainsByClassLoader.remove(loader);
   }
}