  /*
 * JBoss, Home of Professional Open Source.
 * Copyright 2011, Red Hat, Inc., and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

  package org.jboss.as.clustering.subsystem;

import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamWriter;
import java.util.List;
import java.util.Locale;
import java.util.ResourceBundle;

import org.jboss.as.controller.ListAttributeDefinition;
import org.jboss.as.controller.client.helpers.MeasurementUnit;
import org.jboss.as.controller.descriptions.ModelDescriptionConstants;
import org.jboss.as.controller.descriptions.ResourceDescriptionResolver;
import org.jboss.as.controller.operations.validation.AllowedValuesValidator;
import org.jboss.as.controller.operations.validation.MinMaxValidator;
import org.jboss.as.controller.operations.validation.ParameterValidator;
import org.jboss.as.controller.registry.AttributeAccess;
import org.jboss.dmr.ModelNode;

  /**
   * AttributeDefinition suitable for managing LISTs of OBJECTs, which takes into account
   * recursive processing of allowed values and their value types.
   *
   * Date: 13.10.2011
   *
   * @author <a href="mailto:jperkins@redhat.com">James R. Perkins</a>
   * @author Richard Achmatowicz (c) 2012 RedHat Inc.
   */
  public class ObjectListAttributeDefinition extends ListAttributeDefinition {
      private final ObjectTypeAttributeDefinition valueType;

      private ObjectListAttributeDefinition(final String name, final String xmlName, final ObjectTypeAttributeDefinition valueType, final boolean allowNull, final int minSize, final int maxSize, final String[] alternatives, final String[] requires, final AttributeAccess.Flag... flags) {
          super(name, xmlName, allowNull, minSize, maxSize, valueType.getValidator(), alternatives, requires, flags);
          this.valueType = valueType;
      }

      @Override
      public ModelNode addResourceAttributeDescription(ResourceBundle bundle, String prefix, ModelNode resourceDescription) {
          final ModelNode result = super.addResourceAttributeDescription(bundle, prefix, resourceDescription);
          addValueTypeDescription(result, prefix, bundle);
          return result;
      }

      @Override
      public ModelNode addOperationParameterDescription(ResourceBundle bundle, String prefix, ModelNode operationDescription) {
          final ModelNode result = super.addOperationParameterDescription(bundle, prefix, operationDescription);
          addValueTypeDescription(result, prefix, bundle);
          return result;
      }


      @Override
      protected void addValueTypeDescription(final ModelNode node, final ResourceBundle bundle) {
          node.get(ModelDescriptionConstants.VALUE_TYPE, valueType.getName()).set(getValueTypeDescription(false));
      }


      protected void addValueTypeDescription(final ModelNode node, final String prefix, final ResourceBundle bundle) {
          final ModelNode valueTypeDesc = getValueTypeDescription(false);
          valueTypeDesc.get(ModelDescriptionConstants.DESCRIPTION).set(valueType.getAttributeTextDescription(bundle, prefix));
          // node.get(VALUE_TYPE, valueType.getName()).set(valueTypeDesc);
          ModelNode childType = node.get(ModelDescriptionConstants.VALUE_TYPE, valueType.getName()).set(valueTypeDesc);
          // recursively process any child OBJECT attributes
          if (valueType instanceof ObjectTypeAttributeDefinition) {
              ObjectTypeAttributeDefinition.class.cast(valueType).addValueTypeDescription(childType, prefix, bundle);
          }
      }

      @Override
      protected void addAttributeValueTypeDescription(final ModelNode node, final ResourceDescriptionResolver resolver, final Locale locale, final ResourceBundle bundle) {
          final ModelNode valueTypeDesc = getValueTypeDescription(false);
          valueTypeDesc.get(ModelDescriptionConstants.DESCRIPTION).set(resolver.getResourceAttributeValueTypeDescription(getName(), locale, bundle, valueType.getName()));
          node.get(ModelDescriptionConstants.VALUE_TYPE, valueType.getName()).set(valueTypeDesc);
      }

      @Override
      protected void addOperationParameterValueTypeDescription(final ModelNode node, final String operationName, final ResourceDescriptionResolver resolver, final Locale locale, final ResourceBundle bundle) {
          final ModelNode valueTypeDesc = getValueTypeDescription(true);
          valueTypeDesc.get(ModelDescriptionConstants.DESCRIPTION).set(resolver.getOperationParameterValueTypeDescription(operationName, getName(), locale, bundle, valueType.getName()));
          node.get(ModelDescriptionConstants.VALUE_TYPE, valueType.getName()).set(valueTypeDesc);
      }

      @Override
      public void marshallAsElement(final ModelNode resourceModel, final XMLStreamWriter writer) throws XMLStreamException {
          if (resourceModel.hasDefined(getName())) {
              writer.writeStartElement(getXmlName());
              for (ModelNode handler : resourceModel.get(getName()).asList()) {
                  valueType.marshallAsElement(handler, writer);
              }
              writer.writeEndElement();
          }
      }

      private ModelNode getValueTypeDescription(boolean forOperation) {
          final ModelNode result = new ModelNode();
          result.get(ModelDescriptionConstants.TYPE).set(valueType.getType());
          result.get(ModelDescriptionConstants.DESCRIPTION); // placeholder
          result.get(ModelDescriptionConstants.EXPRESSIONS_ALLOWED).set(valueType.isAllowExpression());
          if (forOperation) {
              result.get(ModelDescriptionConstants.REQUIRED).set(!valueType.isAllowNull());
          }
          result.get(ModelDescriptionConstants.NILLABLE).set(isAllowNull());
          final ModelNode defaultValue = valueType.getDefaultValue();
          if (!forOperation && defaultValue != null && defaultValue.isDefined()) {
              result.get(ModelDescriptionConstants.DEFAULT).set(defaultValue);
          }
          MeasurementUnit measurementUnit = valueType.getMeasurementUnit();
          if (measurementUnit != null && measurementUnit != MeasurementUnit.NONE) {
              result.get(ModelDescriptionConstants.UNIT).set(measurementUnit.getName());
          }
          final String[] alternatives = valueType.getAlternatives();
          if (alternatives != null) {
              for (final String alternative : alternatives) {
                  result.get(ModelDescriptionConstants.ALTERNATIVES).add(alternative);
              }
          }
          final String[] requires = valueType.getRequires();
          if (requires != null) {
              for (final String required : requires) {
                  result.get(ModelDescriptionConstants.REQUIRES).add(required);
              }
          }
          final ParameterValidator validator = valueType.getValidator();
          if (validator instanceof MinMaxValidator) {
              MinMaxValidator minMax = (MinMaxValidator) validator;
              Long min = minMax.getMin();
              if (min != null) {
                  switch (valueType.getType()) {
                      case STRING:
                      case LIST:
                      case OBJECT:
                          result.get(ModelDescriptionConstants.MIN_LENGTH).set(min);
                          break;
                      default:
                          result.get(ModelDescriptionConstants.MIN).set(min);
                  }
              }
              Long max = minMax.getMax();
              if (max != null) {
                  switch (valueType.getType()) {
                      case STRING:
                      case LIST:
                      case OBJECT:
                          result.get(ModelDescriptionConstants.MAX_LENGTH).set(max);
                          break;
                      default:
                          result.get(ModelDescriptionConstants.MAX).set(max);
                  }
              }
          }
          if (validator instanceof AllowedValuesValidator) {
              AllowedValuesValidator avv = (AllowedValuesValidator) validator;
              List<ModelNode> allowed = avv.getAllowedValues();
              if (allowed != null) {
                  for (ModelNode ok : allowed) {
                      result.get(ModelDescriptionConstants.ALLOWED).add(ok);
                  }
              }
          }
          return result;
      }

      public static class Builder {
          private final String name;
          private final ObjectTypeAttributeDefinition valueType;
          private String xmlName;
          private boolean allowNull;
          private int minSize;
          private int maxSize;
          private String[] alternatives;
          private String[] requires;
          private AttributeAccess.Flag[] flags;

          public Builder(final String name, final ObjectTypeAttributeDefinition valueType) {
              this.name = name;
              this.valueType = valueType;
          }

          public static Builder of(final String name, final ObjectTypeAttributeDefinition valueType) {
              return new Builder(name, valueType);
          }

          public ObjectListAttributeDefinition build() {
              if (xmlName == null) xmlName = name;
              if (maxSize < 1) maxSize = Integer.MAX_VALUE;
              return new ObjectListAttributeDefinition(name, xmlName, valueType, allowNull, minSize, maxSize, alternatives, requires, flags);
          }

          public Builder setAllowNull(final boolean allowNull) {
              this.allowNull = allowNull;
              return this;
          }

          public Builder setAlternates(final String... alternates) {
              this.alternatives = alternates;
              return this;
          }

          public Builder setFlags(final AttributeAccess.Flag... flags) {
              this.flags = flags;
              return this;
          }

          public Builder setMaxSize(final int maxSize) {
              this.maxSize = maxSize;
              return this;
          }

          public Builder setMinSize(final int minSize) {
              this.minSize = minSize;
              return this;
          }

          public Builder setRequires(final String... requires) {
              this.requires = requires;
              return this;
          }

          public Builder setXmlName(final String xmlName) {
              this.xmlName = xmlName;
              return this;
          }
      }
  }
