/*
 * Decompiled with CFR 0.152.
 */
package org.jboss.as.controller.persistence;

import java.io.File;
import java.io.FilenameFilter;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.List;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.regex.Pattern;
import org.jboss.as.controller.logging.ControllerLogger;
import org.jboss.as.controller.persistence.ConfigurationPersistenceException;
import org.jboss.as.controller.persistence.ConfigurationPersister;
import org.jboss.as.controller.persistence.FilePersistenceUtils;
import org.wildfly.security.manager.WildFlySecurityManager;

public class ConfigurationFile {
    private static final String LAST = "last";
    private static final String INITIAL = "initial";
    private static final String BOOT = "boot";
    private static final String LAST_SUFFIX = "last.xml";
    private static final String INITIAL_SUFFIX = "initial.xml";
    private static final String ORIGINAL_SUFFIX = "boot.xml";
    private static final int CURRENT_HISTORY_LENGTH = 100;
    private static final int HISTORY_DAYS = 30;
    private static final String CURRENT_HISTORY_LENGTH_PROPERTY = "jboss.config.current-history-length";
    private static final String HISTORY_DAYS_PROPERTY = "jboss.config.history-days";
    private static final String TIMESTAMP_STRING = "\\d\\d\\d\\d\\d\\d\\d\\d-\\d\\d\\d\\d\\d\\d\\d\\d\\d";
    private static final Pattern TIMESTAMP_PATTERN = Pattern.compile("\\d\\d\\d\\d\\d\\d\\d\\d-\\d\\d\\d\\d\\d\\d\\d\\d\\d");
    private static final String TIMESTAMP_FORMAT = "yyyyMMdd-HHmmssSSS";
    private static final Pattern VERSION_PATTERN = Pattern.compile("v\\d+");
    private static final Pattern FILE_WITH_VERSION_PATTERN = Pattern.compile("\\S*\\.v\\d+\\.xml");
    private static final Pattern SNAPSHOT_XML = Pattern.compile("\\d\\d\\d\\d\\d\\d\\d\\d-\\d\\d\\d\\d\\d\\d\\d\\d\\d\\S*\\.xml");
    private static final Pattern GENERAL_SNAPSHOT_XML = Pattern.compile("\\S*\\.xml");
    private final AtomicInteger sequence = new AtomicInteger();
    private final AtomicBoolean doneBootup = new AtomicBoolean();
    private final File configurationDir;
    private final String rawFileName;
    private volatile String bootFileName;
    private volatile File bootFile;
    private volatile boolean reloadUsingLast;
    private volatile boolean bootFileReset;
    private volatile String newReloadBootFileName;
    private final File mainFile;
    private final File historyRoot;
    private final File currentHistory;
    private final File snapshotsDirectory;
    private final File serverTempDir;
    private final InteractionPolicy interactionPolicy;
    private volatile File lastFile;
    private final boolean useGit;

    public ConfigurationFile(File configurationDir, String rawName, String name, boolean persistOriginal) {
        this(configurationDir, rawName, name, persistOriginal ? InteractionPolicy.STANDARD : InteractionPolicy.READ_ONLY, false);
    }

    public ConfigurationFile(File configurationDir, String rawName, String name, InteractionPolicy interactionPolicy, boolean useGit) {
        this(configurationDir, rawName, name, interactionPolicy, useGit, null);
    }

    public ConfigurationFile(File configurationDir, String rawName, String name, InteractionPolicy interactionPolicy, boolean useGit, File tmpDir) {
        if (!configurationDir.exists() || !configurationDir.isDirectory()) {
            throw ControllerLogger.ROOT_LOGGER.directoryNotFound(configurationDir.getAbsolutePath());
        }
        assert (rawName != null && rawName.length() > 0);
        this.rawFileName = rawName;
        this.bootFileName = name != null ? name : rawName;
        this.configurationDir = configurationDir;
        this.serverTempDir = tmpDir;
        this.interactionPolicy = interactionPolicy == null ? InteractionPolicy.STANDARD : interactionPolicy;
        this.historyRoot = new File(tmpDir != null && this.interactionPolicy.isReadOnly() && !configurationDir.canWrite() ? tmpDir : configurationDir, rawName.replace('.', '_') + "_history");
        this.currentHistory = new File(this.historyRoot, "current");
        this.snapshotsDirectory = new File(this.historyRoot, "snapshot");
        this.useGit = useGit;
        File file = this.determineMainFile(rawName, name);
        try {
            this.mainFile = file.getCanonicalFile();
        }
        catch (IOException ioe) {
            throw ControllerLogger.ROOT_LOGGER.canonicalMainFileNotFound(ioe, file);
        }
    }

    public boolean useGit() {
        return this.useGit;
    }

    public boolean checkCanFindNewBootFile(String bootFileName) {
        File file = this.determineBootFile(this.configurationDir, bootFileName);
        return file != null && file.exists();
    }

    public synchronized void resetBootFile(boolean reloadUsingLast, String newBootFileName) {
        this.bootFile = null;
        this.bootFileReset = true;
        this.reloadUsingLast = reloadUsingLast;
        this.newReloadBootFileName = newBootFileName;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public File getBootFile() {
        if (this.bootFile == null) {
            ConfigurationFile configurationFile = this;
            synchronized (configurationFile) {
                if (this.bootFile == null) {
                    if (this.bootFileReset) {
                        this.doneBootup.set(false);
                        this.sequence.set(0);
                    }
                    if (this.bootFileReset && !this.interactionPolicy.isReadOnly() && this.newReloadBootFileName == null) {
                        this.bootFile = this.mainFile;
                    } else {
                        String bootFileName = this.bootFileName;
                        if (this.newReloadBootFileName != null) {
                            bootFileName = this.newReloadBootFileName;
                            this.newReloadBootFileName = null;
                        } else if (this.interactionPolicy.isReadOnly() && this.reloadUsingLast) {
                            bootFileName = LAST;
                        }
                        boolean usingRawFile = bootFileName.equals(this.rawFileName);
                        if (usingRawFile) {
                            this.bootFile = this.mainFile;
                        } else {
                            this.bootFile = this.determineBootFile(this.configurationDir, bootFileName);
                            try {
                                this.bootFile = this.bootFile.getCanonicalFile();
                            }
                            catch (IOException ioe) {
                                throw ControllerLogger.ROOT_LOGGER.canonicalBootFileNotFound(ioe, this.bootFile);
                            }
                        }
                        if (!this.bootFile.exists()) {
                            if (!usingRawFile && (this.bootFileReset || this.interactionPolicy.isRequireExisting())) {
                                throw ControllerLogger.ROOT_LOGGER.fileNotFound(this.bootFile.getAbsolutePath());
                            }
                            if (!this.bootFileReset && !this.interactionPolicy.isRequireExisting()) {
                                ConfigurationFile.createBootFile(this.bootFile);
                            }
                        } else if (!this.bootFileReset) {
                            if (this.interactionPolicy.isRejectExisting() && this.bootFile.length() > 0L) {
                                throw ControllerLogger.ROOT_LOGGER.rejectEmptyConfig(this.bootFile.getAbsolutePath());
                            }
                            if (this.interactionPolicy.isRemoveExisting() && this.bootFile.length() > 0L) {
                                if (!this.bootFile.delete()) {
                                    throw ControllerLogger.ROOT_LOGGER.cannotDelete(this.bootFile.getAbsoluteFile());
                                }
                                ConfigurationFile.createBootFile(this.bootFile);
                            }
                        }
                    }
                }
            }
        }
        return this.bootFile;
    }

    public InteractionPolicy getInteractionPolicy() {
        return this.interactionPolicy;
    }

    private File determineMainFile(String rawName, String name) {
        assert (rawName != null);
        String mainName = null;
        if (name == null) {
            mainName = rawName;
        } else if (name.equals(LAST) || name.equals(INITIAL) || name.equals(BOOT)) {
            mainName = this.findMainFileFromBackupSuffix(this.historyRoot, name);
        } else if (VERSION_PATTERN.matcher(name).matches()) {
            mainName = this.findMainFileFromBackupSuffix(this.currentHistory, name);
        }
        if (mainName == null) {
            mainName = this.findMainFileFromSnapshotPrefix(name);
        }
        if (mainName == null) {
            File absoluteFile;
            File directoryFile = new File(this.configurationDir, name);
            if (directoryFile.exists()) {
                mainName = this.stripPrefixSuffix(name);
            } else if (this.interactionPolicy.isReadOnly() && (absoluteFile = new File(name)).exists()) {
                return absoluteFile;
            }
        }
        if (mainName == null && !this.interactionPolicy.isRequireExisting()) {
            mainName = this.stripPrefixSuffix(name);
        }
        if (mainName != null) {
            return new File(this.configurationDir, new File(mainName).getName());
        }
        throw ControllerLogger.ROOT_LOGGER.mainFileNotFound(name != null ? name : rawName, this.configurationDir);
    }

    private String findMainFileFromBackupSuffix(File searchDir, String backupType) {
        final String suffix = "." + backupType + ".xml";
        File[] files = null;
        if (searchDir.exists() && searchDir.isDirectory()) {
            files = searchDir.listFiles(new FilenameFilter(){

                @Override
                public boolean accept(File dir, String name) {
                    return name.endsWith(suffix);
                }
            });
        }
        if (files == null || files.length == 0) {
            throw ControllerLogger.ROOT_LOGGER.configurationFileNotFound(suffix, searchDir);
        }
        if (files.length > 1) {
            throw ControllerLogger.ROOT_LOGGER.ambiguousConfigurationFiles(backupType, searchDir, suffix);
        }
        String matchName = files[0].getName();
        if (matchName.equals(suffix)) {
            throw ControllerLogger.ROOT_LOGGER.configurationFileNameNotAllowed(backupType);
        }
        String prefix = matchName.substring(0, matchName.length() - suffix.length());
        return prefix + ".xml";
    }

    private String findMainFileFromSnapshotPrefix(final String prefix) {
        File[] files = null;
        if (this.snapshotsDirectory.exists() && this.snapshotsDirectory.isDirectory()) {
            files = this.snapshotsDirectory.listFiles(new FilenameFilter(){

                @Override
                public boolean accept(File dir, String name) {
                    return name.startsWith(prefix) && SNAPSHOT_XML.matcher(name).find();
                }
            });
        }
        if (files == null || files.length == 0) {
            return null;
        }
        if (files.length > 1) {
            throw ControllerLogger.ROOT_LOGGER.ambiguousConfigurationFiles(prefix, this.snapshotsDirectory, prefix);
        }
        String matchName = files[0].getName();
        return matchName.substring(TIMESTAMP_FORMAT.length());
    }

    private String stripPrefixSuffix(String name) {
        if (SNAPSHOT_XML.matcher(name).matches()) {
            name = name.substring(TIMESTAMP_FORMAT.length());
        }
        if (FILE_WITH_VERSION_PATTERN.matcher(name).matches()) {
            int last = name.lastIndexOf(118);
            name = name.substring(0, last) + "xml";
        } else if (name.endsWith(LAST_SUFFIX)) {
            name = name.substring(0, name.length() - LAST_SUFFIX.length()) + "xml";
        } else if (name.endsWith(ORIGINAL_SUFFIX)) {
            name = name.substring(0, name.length() - ORIGINAL_SUFFIX.length()) + "xml";
        } else if (name.endsWith(INITIAL_SUFFIX)) {
            name = name.substring(0, name.length() - INITIAL_SUFFIX.length()) + "xml";
        }
        return name;
    }

    private File determineBootFile(File configurationDir, String name) {
        File result;
        File directoryFile = new File(configurationDir, name);
        if (name.equals(LAST) || name.equals(INITIAL) || name.equals(BOOT)) {
            result = this.addSuffixToFile(new File(this.historyRoot, this.mainFile.getName()), name);
        } else if (VERSION_PATTERN.matcher(name).matches()) {
            result = this.getVersionedFile(this.mainFile, name);
        } else {
            result = this.findSnapshotWithPrefix(name, false);
            if (result == null) {
                File absoluteFile;
                if (directoryFile.exists()) {
                    result = directoryFile;
                } else if (this.interactionPolicy.isReadOnly() && (absoluteFile = new File(name)).exists()) {
                    result = absoluteFile;
                }
            }
        }
        if (result == null) {
            result = directoryFile;
        }
        return result;
    }

    private static void createBootFile(File toCreate) {
        IOException cause = null;
        try {
            if (toCreate.createNewFile()) {
                return;
            }
        }
        catch (IOException e) {
            cause = e;
        }
        throw ControllerLogger.ROOT_LOGGER.cannotCreateEmptyConfig(toCreate.getAbsolutePath(), cause);
    }

    public File getMainFile() {
        return this.mainFile;
    }

    public File getConfigurationDir() {
        return this.configurationDir;
    }

    File getConfigurationTmpDir() {
        return this.serverTempDir;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void successfulBoot() throws ConfigurationPersistenceException {
        ConfigurationFile configurationFile = this;
        synchronized (configurationFile) {
            File copySource;
            if (this.doneBootup.get()) {
                return;
            }
            if (!this.interactionPolicy.isReadOnly()) {
                copySource = this.mainFile;
            } else {
                copySource = FilePersistenceUtils.isParentFolderWritable(this.mainFile) ? new File(this.mainFile.getParentFile(), this.mainFile.getName() + ".boot") : new File(this.serverTempDir, this.mainFile.getName() + ".boot");
                FilePersistenceUtils.deleteFile(copySource);
            }
            try {
                if (!this.bootFile.equals(copySource)) {
                    FilePersistenceUtils.copyFile(this.bootFile, copySource);
                }
                this.createHistoryDirectory();
                File historyBase = new File(this.historyRoot, this.mainFile.getName());
                this.lastFile = this.addSuffixToFile(historyBase, LAST);
                File boot = this.addSuffixToFile(historyBase, BOOT);
                File initial = this.addSuffixToFile(historyBase, INITIAL);
                if (!initial.exists()) {
                    FilePersistenceUtils.copyFile(copySource, initial);
                }
                FilePersistenceUtils.copyFile(copySource, this.lastFile);
                FilePersistenceUtils.copyFile(copySource, boot);
            }
            catch (IOException e) {
                throw ControllerLogger.ROOT_LOGGER.failedToCreateConfigurationBackup(e, this.bootFile);
            }
            finally {
                if (this.interactionPolicy.isReadOnly()) {
                    try {
                        FilePersistenceUtils.deleteFile(copySource);
                    }
                    catch (Exception exception) {}
                }
            }
            this.doneBootup.set(true);
        }
    }

    void backup() throws ConfigurationPersistenceException {
        if (!this.doneBootup.get()) {
            return;
        }
        try {
            if (!this.interactionPolicy.isReadOnly()) {
                this.moveFile(this.mainFile, this.getVersionedFile(this.mainFile));
            } else {
                this.moveFile(this.lastFile, this.getVersionedFile(this.mainFile));
            }
            int seq = this.sequence.get();
            int currentHistoryLength = this.getInteger(CURRENT_HISTORY_LENGTH_PROPERTY, 100, 0);
            if (seq > currentHistoryLength) {
                File delete;
                for (int k = seq - currentHistoryLength; k > 0 && (delete = this.getVersionedFile(this.mainFile, k)).exists(); --k) {
                    delete.delete();
                }
            }
        }
        catch (IOException e) {
            throw ControllerLogger.ROOT_LOGGER.failedToBackup(e, this.mainFile);
        }
    }

    void commitTempFile(File temp) throws ConfigurationPersistenceException {
        if (!this.doneBootup.get()) {
            return;
        }
        if (!this.interactionPolicy.isReadOnly()) {
            FilePersistenceUtils.moveTempFileToMain(temp, this.mainFile);
        } else {
            FilePersistenceUtils.moveTempFileToMain(temp, this.lastFile);
        }
    }

    void fileWritten() throws ConfigurationPersistenceException {
        if (!this.doneBootup.get() || this.interactionPolicy.isReadOnly()) {
            return;
        }
        try {
            FilePersistenceUtils.copyFile(this.mainFile, this.lastFile);
        }
        catch (IOException e) {
            throw ControllerLogger.ROOT_LOGGER.failedToBackup(e, this.mainFile);
        }
    }

    private void moveFile(File file, File backup) throws IOException {
        Files.move(file.toPath(), backup.toPath(), StandardCopyOption.REPLACE_EXISTING);
    }

    String snapshot(String prefix, String comment) throws ConfigurationPersistenceException {
        String sanitizedComment = FilePersistenceUtils.sanitizeFileName(comment);
        String fileName = sanitizedComment == null || sanitizedComment.isEmpty() ? this.mainFile.getName() : sanitizedComment + "-" + this.mainFile.getName();
        String name = prefix == null || prefix.isEmpty() ? ConfigurationFile.getTimeStamp(new Date()) + fileName : prefix + fileName;
        File snapshot = new File(this.snapshotsDirectory, name);
        File source = this.interactionPolicy.isReadOnly() ? this.lastFile : this.mainFile;
        try {
            FilePersistenceUtils.copyFile(source, snapshot);
        }
        catch (IOException e) {
            throw ControllerLogger.ROOT_LOGGER.failedToTakeSnapshot(e, source, snapshot);
        }
        return snapshot.toString();
    }

    ConfigurationPersister.SnapshotInfo listSnapshots() {
        return new BackupSnapshotInfo();
    }

    void deleteSnapshot(String prefix) {
        if (prefix.equals("all")) {
            if (this.snapshotsDirectory.exists() && this.snapshotsDirectory.isDirectory()) {
                for (String curr : this.snapshotsDirectory.list()) {
                    new File(this.snapshotsDirectory, curr).delete();
                }
            }
        } else {
            this.findSnapshotWithPrefix(prefix, true).delete();
        }
    }

    private File findSnapshotWithPrefix(String prefix, boolean errorIfNoFiles) {
        ArrayList<String> names = new ArrayList<String>();
        if (this.snapshotsDirectory.exists() && this.snapshotsDirectory.isDirectory()) {
            for (String curr : this.snapshotsDirectory.list()) {
                if (!curr.startsWith(prefix)) continue;
                names.add(curr);
            }
        }
        if (names.isEmpty() && errorIfNoFiles) {
            throw ControllerLogger.ROOT_LOGGER.fileNotFoundWithPrefix(prefix, this.snapshotsDirectory.getAbsolutePath());
        }
        if (names.size() > 1) {
            throw ControllerLogger.ROOT_LOGGER.ambiguousName(prefix, this.snapshotsDirectory.getAbsolutePath(), names);
        }
        return !names.isEmpty() ? new File(this.snapshotsDirectory, (String)names.get(0)) : null;
    }

    private void createHistoryDirectory() throws IOException {
        this.mkdir(this.historyRoot);
        this.mkdir(this.snapshotsDirectory);
        if (this.currentHistory.exists()) {
            if (!this.currentHistory.isDirectory()) {
                throw ControllerLogger.ROOT_LOGGER.notADirectory(this.currentHistory.getAbsolutePath());
            }
            Date date = new Date();
            File[] currentHistoryFiles = this.currentHistory.listFiles();
            if (currentHistoryFiles != null && currentHistoryFiles.length > 0) {
                String backupName = ConfigurationFile.getTimeStamp(date);
                File old = new File(this.historyRoot, backupName);
                if (!ConfigurationFile.forcedMove(this.currentHistory.toPath(), old.toPath())) {
                    if (old.exists()) {
                        date = new Date(date.getTime() + 100L);
                        backupName = ConfigurationFile.getTimeStamp(date);
                        old = new File(this.historyRoot, backupName);
                        if (!ConfigurationFile.forcedMove(this.currentHistory.toPath(), old.toPath())) {
                            ControllerLogger.ROOT_LOGGER.couldNotCreateHistoricalBackup(this.currentHistory.getAbsolutePath());
                        }
                    } else {
                        ControllerLogger.ROOT_LOGGER.couldNotCreateHistoricalBackup(this.currentHistory.getAbsolutePath());
                    }
                }
            }
            int historyDays = this.getInteger(HISTORY_DAYS_PROPERTY, 30, 0);
            String cutoffFileName = ConfigurationFile.getTimeStamp(this.subtractDays(date, historyDays));
            for (String name : this.historyRoot.list()) {
                if (name.length() != cutoffFileName.length() || !TIMESTAMP_PATTERN.matcher(name).matches() || name.compareTo(cutoffFileName) >= 0) continue;
                this.deleteRecursive(new File(this.historyRoot, name));
            }
        }
        this.currentHistory.mkdir();
        if (!this.currentHistory.exists()) {
            throw ControllerLogger.ROOT_LOGGER.cannotCreate(this.currentHistory.getAbsolutePath());
        }
    }

    private int getInteger(String name, int defaultValue, int minimalValue) {
        int retVal = this.getInteger(name, defaultValue);
        return retVal < minimalValue ? defaultValue : retVal;
    }

    private int getInteger(String name, int defaultValue) {
        String val = WildFlySecurityManager.getPropertyPrivileged(name, null);
        try {
            return val == null ? defaultValue : Integer.parseInt(val);
        }
        catch (NumberFormatException ignored) {
            return defaultValue;
        }
    }

    private void deleteRecursive(File file) {
        String[] files;
        if (file.isDirectory() && (files = file.list()) != null) {
            for (String name : files) {
                this.deleteRecursive(new File(file, name));
            }
        }
        if (!file.delete()) {
            ControllerLogger.ROOT_LOGGER.cannotDeleteFileOrDirectory(file);
        }
    }

    private File getVersionedFile(File file) {
        return this.getVersionedFile(file, this.sequence.incrementAndGet());
    }

    private File getVersionedFile(File file, int i) {
        return this.addSuffixToFile(new File(this.currentHistory, file.getName()), "v" + i);
    }

    private File getVersionedFile(File file, String versionString) {
        return this.addSuffixToFile(new File(this.currentHistory, file.getName()), versionString);
    }

    private File addSuffixToFile(File file, String suffix) {
        Path path = file.toPath();
        String fileName = path.getFileName().toString();
        int index = fileName.lastIndexOf(46);
        if (index == -1) {
            return path.resolveSibling(fileName + '.' + suffix).toFile();
        }
        StringBuilder sb = new StringBuilder();
        sb.append(fileName.substring(0, index));
        sb.append('.');
        sb.append(suffix);
        sb.append(fileName.substring(index));
        return path.resolveSibling(sb.toString()).toFile();
    }

    private Date subtractDays(Date date, int days) {
        GregorianCalendar calendar = new GregorianCalendar();
        calendar.setTime(date);
        int doy = calendar.get(6);
        calendar.set(6, doy - days);
        return calendar.getTime();
    }

    private static String getTimeStamp(Date date) {
        SimpleDateFormat sfd = new SimpleDateFormat(TIMESTAMP_FORMAT);
        return sfd.format(date);
    }

    private File mkdir(File dir) {
        if (!dir.exists()) {
            if (!dir.mkdir()) {
                throw ControllerLogger.ROOT_LOGGER.cannotCreate(this.historyRoot.getAbsolutePath());
            }
        } else if (!dir.isDirectory()) {
            throw ControllerLogger.ROOT_LOGGER.notADirectory(dir.getAbsolutePath());
        }
        return dir;
    }

    private static boolean forcedMove(Path from, Path to) {
        try {
            Files.move(from, to, StandardCopyOption.REPLACE_EXISTING);
            return true;
        }
        catch (IOException e) {
            ControllerLogger.ROOT_LOGGER.cannotRename(e, from, to);
            return false;
        }
    }

    private class BackupSnapshotInfo
    implements ConfigurationPersister.SnapshotInfo {
        final ArrayList<String> names = new ArrayList();

        public BackupSnapshotInfo() {
            for (String name : ConfigurationFile.this.snapshotsDirectory.list(new FilenameFilter(){

                @Override
                public boolean accept(File dir, String name) {
                    return GENERAL_SNAPSHOT_XML.matcher(name).matches();
                }
            })) {
                this.names.add(name);
            }
        }

        @Override
        public String getSnapshotDirectory() {
            return ConfigurationFile.this.snapshotsDirectory.getAbsolutePath();
        }

        @Override
        public List<String> names() {
            return this.names;
        }
    }

    public static enum InteractionPolicy {
        STANDARD(true, false, false, false),
        DISCARD(false, false, true, false),
        NEW(false, true, false, false),
        READ_ONLY(true, false, false, true);

        private final boolean requireExisting;
        private final boolean rejectExisting;
        private final boolean removeExisting;
        private final boolean readOnly;

        private InteractionPolicy(boolean requireExisting, boolean rejectExisting, boolean removeExisting, boolean readOnly) {
            this.requireExisting = requireExisting;
            this.rejectExisting = rejectExisting;
            this.removeExisting = removeExisting;
            this.readOnly = readOnly;
        }

        public boolean isReadOnly() {
            return this.readOnly;
        }

        private boolean isRequireExisting() {
            return this.requireExisting;
        }

        private boolean isRejectExisting() {
            return this.rejectExisting;
        }

        private boolean isRemoveExisting() {
            return this.removeExisting;
        }
    }
}

