/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2009, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
  *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.bootstrap.api.as.config;

import java.net.URL;

import org.jboss.bootstrap.api.config.ServerConfig;

/**
 * JBossASBasedServerConfig
 * 
 * Contract for the configuration of AS-based Servers, 
 * may be extended for a direct AS ServerConfig
 *
 * @author <a href="mailto:andrew.rubinger@jboss.org">ALR</a>
 * @version $Revision: $
 */
public interface JBossASBasedServerConfig<T extends JBossASBasedServerConfig<T>> extends ServerConfig<T>
{
   //-------------------------------------------------------------------------------||
   // Constants --------------------------------------------------------------------||
   //-------------------------------------------------------------------------------||

   /**
    * Constant that holds the name of the system or
    * configuration property 
    */
   String PROP_KEY_JBOSSAS_BIND_ADDRESS = "jboss.bind.address";

   /**
    * Constant that holds the name of the system or configuration 
    * property which specifies the name of the server which will be used to
    * calculate the servers home directory and url.
    */
   String PROP_KEY_JBOSSAS_SERVER_NAME = "jboss.server.name";

   /**
    * Constant that holds the name of the environment
    * variable which specifies the root of the AS
    * installation
    */
   String ENV_VAR_JBOSSAS_HOME = "JBOSS_HOME";

   /**
    * Constant that holds the name of the system or configuration 
    * property which specifies the root of the AS
    * installation
    */
   String PROP_KEY_JBOSSAS_HOME = "jboss.home";

   /**
    * Constant that holds the name of the system property
    * which specifies the home directory for JBossAS.
    */
   String PROP_KEY_JBOSSAS_HOME_DIR = "jboss.home.dir";

   /**
    * Constant that holds the name of the system property
    * which specifies the home URL for JBossAS.
    */
   String PROP_KEY_JBOSSAS_HOME_URL = "jboss.home.url";

   /**
    * Constant that holds the name of the system property
    * for specifying the URL where JBoss will read library files
    * from.
    */
   String PROP_KEY_JBOSSAS_BOOT_LIBRARY_URL = "jboss.lib.url";

   /**
    * Constant that holds the name of the system property
    * for specifying the base directory for calculating server
    * home directories.
    */
   String PROP_KEY_JBOSSAS_SERVER_BASE_DIR = "jboss.server.base.dir";

   /**
    * Constant that holds the name of the system property
    * for specifying the base URL for calculating server
    * home URLs.
    */
   String PROP_KEY_JBOSSAS_SERVER_BASE_URL = "jboss.server.base.url";

   /**
    * Constant that holds the name of the system property
    * for specifying the server home directory for JBoss.
    */
   String PROP_KEY_JBOSSAS_SERVER_HOME_DIR = "jboss.server.home.dir";

   /**
    * Constant that holds the name of the system property
    * for specifying the server home URL for JBoss.
    */
   String PROP_KEY_JBOSSAS_SERVER_HOME_URL = "jboss.server.home.url";

   /**
    * Constant that holds the name of the system property
    * for specifying the base URL for files and directories
    * common to different server configurations
    */
   String PROP_KEY_JBOSSAS_COMMON_BASE_URL = "jboss.common.base.url";

   /**
    * Constant that holds the name of the system property
    * for specifying a library directory URL that points to libraries
    * shared by the various server configurations.
    */
   String PROP_KEY_JBOSSAS_COMMON_LIBRARY_URL = "jboss.common.lib.url";

   /**
    * Constant that holds the name of the system property
    * for specifying the server log directory for JBoss.
    */
   String PROP_KEY_JBOSSAS_SERVER_LOG_DIR = "jboss.server.log.dir";

   /**
    * Constant that holds the name of the system property
    * for specifying the server configuration URL.
    */
   String PROP_KEY_JBOSSAS_SERVER_CONF_URL = "jboss.server.config.url";

   /**
    * Constant that holds the name of the system property
    * for specifying the URL where JBoss will read server specific
    * library files from.
    */
   String PROP_KEY_JBOSSAS_SERVER_LIBRARY_URL = "jboss.server.lib.url";

   /**
    * Constant that holds the name of the system property
    * for specifying the directory which JBoss will use for
    * persistent data file storage.
    */
   String PROP_KEY_JBOSSAS_SERVER_DATA_DIR = "jboss.server.data.dir";

   /**
    * Constant that holds the name of the system property
    * for specifying the directory which JBoss will use for
    * temporary file storage.
    */
   String PROP_KEY_JBOSSAS_SERVER_TEMP_DIR = "jboss.server.temp.dir";

   /** 
    * Constant that holds the name of the system property
    * for specifying the partition name for this instance in clustering
    */
   String PROP_KEY_JBOSSAS_PARTITION_NAME = "jboss.partition.name";

   /** 
    * Constant that holds the name of the system property
    * for specifying the UDP Group for this instance in clustering
    */
   String PROP_KEY_JBOSSAS_PARTITION_UDP_GROUP = "jboss.partition.udpGroup";

   /** 
    * Constant that holds the name of the system property
    * for specifying the UDP Port for this instance in clustering
    */
   String PROP_KEY_JBOSSAS_PARTITION_UDP_PORT = "jboss.jgroups.udp.mcast_port";

   /** 
    * Constant that holds the name of the system property
    * designating whether to load native libraries
    */
   String PROP_KEY_JBOSSAS_NATIVE_LOAD = "jboss.native.load";

   /** 
    * Constant that holds the location of native libraries 
    */
   String PROP_KEY_JBOSSAS_NATIVE_DIR = "jboss.native.dir";

   /**
    * Constant that holds the name of the system property for specifying
    * whether or not to use as the main jboss server the MBeanServer returned
    * from ManagementFactory.getPlatformMBeanServer(), when running under jdk1.5+
    * 
    * <p>If not set then jboss will instantiate its own MBeanServer
    */
   String PROP_KEY_JBOSSAS_PLATFORM_MBEANSERVER = "jboss.platform.mbeanserver";

   //-------------------------------------------------------------------------------||
   // Contracts --------------------------------------------------------------------||
   //-------------------------------------------------------------------------------||

   /**
    * Obtains the bind address to be used for the Server.
    * 
    * @return
    */
   String getBindAddress();

   /**
    * Sets the bind address for the server
    * 
    * @param bindAddress
    * @return
    */
   T bindAddress(String bindAddress);

   /**
    * Obtains the name of the server configuration. 
    *  
    * @return
    */
   String getServerName();

   /**
    * Sets the name of the server configuration 
    * 
    * @param serverName
    * @return
    */
   T serverName(String serverName);

   /**
    * Obtains the $JBOSS_HOME, the path to the
    * root location of the JBossAS
    * installation.
    * 
    * @return
    */
   URL getJBossHome();

   /**
    * Sets $JBOSS_HOME, the path to the root location 
    * of the JBossAS installation.
    * 
    * @param jbossHome The location to $JBOSS_HOME
    * @return This configuration
    * @throws IllegalArgumentException If the specified argument 
    *    is non-null and cannot be constructed into a URL
    */
   T jbossHome(String jbossHome) throws IllegalArgumentException;

   /**
    * Sets $JBOSS_HOME, the path to the root location 
    * of the JBossAS installation.
    * 
    * @param jbossHome The location to $JBOSS_HOME
    * @return This configuration
    */
   T jbossHome(URL jbossHome);

   /**
    * Obtains the location of the JBossAS
    * Boot Libraries
    * 
    * @return
    */
   URL getBootLibraryLocation();

   /**
    * Sets the location of the JBossAS
    * Boot Libraries
    * 
    * @param bootLibraryLocation
    * @return
    */
   T bootLibraryLocation(URL bootLibraryLocation);

   /**
    * Sets the location of the JBossAS
    * Boot Libraries
    * 
    * @param bootLibraryLocation
    * @return
    * @throws IllegalArgumentException If the specified argument 
    *    is non-null and cannot be constructed into a URL
    */
   T bootLibraryLocation(String bootLibraryLocation) throws IllegalArgumentException;

   /**
    * Obtains the location of the server base, from
    * which server homes (by default) decend
    * 
    * @return
    */
   URL getServerBaseLocation();

   /**
    * Sets the location of the server
    * base, from which server homes (by default) 
    * decend
    * 
    * @param serverBaseLocation
    * @return
    */
   T serverBaseLocation(URL serverBaseLocation);

   /**
    * Sets the location of the server
    * base, from which server homes (by default) 
    * decend
    * 
    * @param serverBaseLocation
    * @return
    * @throws IllegalArgumentException If the specified argument 
    *    is non-null and cannot be constructed into a URL
    */
   T serverBaseLocation(String serverBaseLocation) throws IllegalArgumentException;

   /**
    * Obtains the location of the server home
    * 
    * @return
    */
   URL getServerHomeLocation();

   /**
    * Sets the location of the server home
    * 
    * @param serverBaseLocation
    * @return
    */
   T serverHomeLocation(URL serverHomeLocation);

   /**
    * Sets the location of the server home
    * 
    * @param serverHomeLocation
    * @return
    * @throws IllegalArgumentException If the specified argument 
    *    is non-null and cannot be constructed into a URL
    */
   T serverHomeLocation(String serverHomeLocation) throws IllegalArgumentException;

   /**
    * Obtains the location of the common base
    * 
    * @return
    */
   URL getCommonBaseLocation();

   /**
    * Sets the location of the common base
    * 
    * @param serverBaseLocation
    * @return
    */
   T commonBaseLocation(URL commonBaseLocation);

   /**
    * Sets the location of the common base
    * 
    * @param commonBaseLocation
    * @return
    * @throws IllegalArgumentException If the specified argument 
    *    is non-null and cannot be constructed into a URL
    */
   T commonBaseLocation(String commonBaseLocation) throws IllegalArgumentException;

   /**
    * Obtains the location of the common lib
    * 
    * @return
    */
   URL getCommonLibLocation();

   /**
    * Sets the location of the common lib
    * 
    * @param commonLibLocation
    * @return
    */
   T commonLibLocation(URL commonLibLocation);

   /**
    * Sets the location of the common lib
    * 
    * @param commonLibLocation
    * @return
    * @throws IllegalArgumentException If the specified argument 
    *    is non-null and cannot be constructed into a URL
    */
   T commonLibLocation(String commonLibLocation) throws IllegalArgumentException;

   /**
    * Obtains the location in which the server logs reside
    * 
    * @return
    */
   URL getServerLogLocation();

   /**
    * Sets the location in which the server logs reside
    * 
    * @param serverLogLocation
    * @return
    */
   T serverLogLocation(URL serverLogLocation);

   /**
    * Sets the location in which the server logs reside
    * 
    * @param serverLogLocation
    * @return
    * @throws IllegalArgumentException If the specified argument 
    *    is non-null and cannot be constructed into a URL
    */
   T serverLogLocation(String serverLogLocation) throws IllegalArgumentException;

   /**
    * Obtains the location in which the server configuration resides
    * 
    * @return
    */
   URL getServerConfLocation();

   /**
    * Sets the location in which the server configuration resides
    * 
    * @param serverConfLocation
    * @return
    */
   T serverConfLocation(URL serverConfLocation);

   /**
    * Sets the location in which the server configuration resides
    * 
    * @param serverConfLocation
    * @return
    * @throws IllegalArgumentException If the specified argument 
    *    is non-null and cannot be constructed into a URL
    */
   T serverConfLocation(String serverConfLocation) throws IllegalArgumentException;

   /**
    * Obtains the location in which the server libraries reside
    * 
    * @return
    */
   URL getServerLibLocation();

   /**
    * Sets the location in which the server libraries reside
    * 
    * @param serverLibLocation
    * @return
    */
   T serverLibLocation(URL serverLibLocation);

   /**
    * Sets the location in which the server libraries reside
    * 
    * @param serverLibLocation
    * @return
    * @throws IllegalArgumentException If the specified argument 
    *    is non-null and cannot be constructed into a URL
    */
   T serverLibLocation(String serverLibLocation) throws IllegalArgumentException;

   /**
    * Obtains the location in which the server persistent data resides
    * 
    * @return
    */
   URL getServerDataLocation();

   /**
    * Sets the location in which the server persistent data resides
    * 
    * @param serverDataLocation
    * @return
    */
   T serverDataLocation(URL serverDataLocation);

   /**
    * Sets the location in which the server persistent data resides
    * 
    * @param serverDataLocation
    * @return
    * @throws IllegalArgumentException If the specified argument 
    *    is non-null and cannot be constructed into a URL
    */
   T serverDataLocation(String serverDataLocation) throws IllegalArgumentException;

   /**
    * Obtains the location in which the server temp data resides
    * 
    * @return
    */
   URL getServerTempLocation();

   /**
    * Sets the location in which the server temp data resides
    * 
    * @param serverTempLocation
    * @return
    */
   T serverTempLocation(URL serverTempLocation);

   /**
    * Sets the location in which the server temp data resides
    * 
    * @param serverTempLocation
    * @return
    * @throws IllegalArgumentException If the specified argument 
    *    is non-null and cannot be constructed into a URL
    */
   T serverTempLocation(String serverTempLocation) throws IllegalArgumentException;

   /**
    * Obtains the partition name for this instance in clustering
    * 
    * @return
    */
   String getPartitionName();

   /**
    * Sets the partition name for this instance in clustering
    * 
    * @param partitionName
    * @return
    */
   T partitionName(String partitionName);

   /**
    * Obtains the UDP Group for this instance in clustering
    * 
    * @return
    */
   String getUdpGroup();

   /**
    * Sets the UDP Group for this instance in clustering
    * 
    * @param udpGroup
    * @return
    */
   T udpGroup(String udpGroup);

   /**
    * Obtains the JGroups UDP Multicast port for use 
    * in clustering
    * 
    * @return
    */
   Integer getUdpPort();

   /**
    * Sets the JGroups UDP Multicast port for use 
    * in clustering
    * 
    * @param udpPort The port, or null to use the default
    * @return
    */
   T udpPort(Integer udpPort);

   /**
    * Determines whether or not to load the native libraries
    * 
    * @return
    */
   Boolean isLoadNative();

   /**
    * Sets whether or not to load the native libraries
    * 
    * @param loadNative true or false, or null to use the default
    * @return
    */
   T loadNative(Boolean loadNative);

   /**
    * Obtains the location of the JBossAS
    * Native Libraries
    * 
    * @return
    */
   URL getNativeLibraryLocation();

   /**
    * Sets the location of the JBossAS
    * Native Libraries
    * 
    * @param nativeLibraryLocation
    * @return
    */
   T nativeLibraryLocation(URL nativeLibraryLocation);

   /**
    * Sets the location of the JBossAS
    * Native Libraries
    * 
    * @param nativeLibraryLocation
    * @return
    * @throws IllegalArgumentException If the specified argument 
    *    is non-null and cannot be constructed into a URL
    */
   T nativeLibraryLocation(String nativeLibraryLocation) throws IllegalArgumentException;

   /**
    * Determines whether or not to use as the main jboss server the MBeanServer returned
    * from ManagementFactory.getPlatformMBeanServer(), when running under jdk1.5+
    * 
    * @return
    */
   Boolean isUsePlatformMBeanServer();

   /**
    * Sets whether or not to use as the main jboss server the MBeanServer returned
    * from ManagementFactory.getPlatformMBeanServer(), when running under jdk1.5+
    * 
    * @param usePlatformMBeanServer
    * @return
    */
   T usePlatformMBeanServer(Boolean usePlatformMBeanServer);

}
