/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2009, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.bootstrap.api.factory;

/**
 * GenericFactory
 * 
 * Factory for creating generic instances
 *
 * @author <a href="mailto:andrew.rubinger@jboss.org">ALR</a>
 * @version $Revision: $
 */
class GenericFactory
{
   //-------------------------------------------------------------------------------------||
   // Constructor ------------------------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   /**
    * Constructor 
    */
   private GenericFactory()
   {
      // No external creation of instances
   }

   //-------------------------------------------------------------------------------------||
   // Functional Methods -----------------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   /**
    * Creates a instance from the specified implementation class name, using the
    * specified ClassLoader.  The returned instance will additionally be created using
    * the ClassLoader denoted. 
    * 
    * @param <T> The expected type
    * @param implClassName The fully-qualified name of the implementation class
    * @param cl The ClassLoader to use
    * @param expectedType The Class to which the resultant object should be assignable 
    * @return The newly-created instance
    * @throws IllegalArgumentException If the expected type is null,
    *       ClassLoader is null, implementation class 
    *       name is null, blank, can not be found on the ClassLoader, 
    *       or is not assignable to the expected type.
    * @throws Exception If some error occurred in constructing the instance
    */
   final static <T> T createInstance(final String implClassName, final ClassLoader cl, final Class<T> expectedType)
         throws IllegalArgumentException, Exception
   {
      // Precondition checks
      if (implClassName == null || implClassName.length() == 0)
      {
         throw new IllegalArgumentException("Implementation class must be specified");
      }
      if (cl == null)
      {
         throw new IllegalArgumentException(ClassLoader.class.getSimpleName() + " must be specified");
      }
      if (expectedType == null)
      {
         throw new IllegalArgumentException("Expected type must be specified");
      }

      // Get the implementation class
      Class<?> implClass = null;
      try
      {
         implClass = Class.forName(implClassName, false, cl);
      }
      catch (ClassNotFoundException cnfe)
      {
         throw new IllegalArgumentException("Specified implementation class could not be found: " + implClassName
               + " on " + ClassLoader.class.getSimpleName() + " " + cl, cnfe);
      }

      // Make a new instance
      final Object obj = SecurityActions.newInstance(implClass, cl);

      // Return
      try
      {
         return expectedType.cast(obj);
      }
      catch (ClassCastException cce)
      {
         // Rethrow as new CCE with some context
         final ClassLoader implCl = implClass.getClassLoader();
         final ClassLoader expectedTypeCl = expectedType.getClassLoader();
         throw new ClassCastException("Specified implementation class, " + implClassName + ", resolved to " + implClass
               + " on " + implCl + " is not of type " + expectedType + " from " + ClassLoader.class.getSimpleName()
               + ": " + (expectedTypeCl == null ? "[BOOTSTRAP CL]" : expectedTypeCl));
      }
   }
}
