/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2009, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.bootstrap.api.factory;

import java.security.AccessController;
import java.security.PrivilegedAction;
import java.security.PrivilegedActionException;
import java.security.PrivilegedExceptionAction;

/**
 * SecurityActions
 * 
 * Utility class for security actions, so as to provide
 * some centralization without leaking privileged actions
 *
 * @author <a href="mailto:andrew.rubinger@jboss.org">ALR</a>
 * @version $Revision: $
 */
class SecurityActions
{

   //-------------------------------------------------------------------------------------||
   // Constructor ------------------------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   /**
    * In place to prevent instanciation 
    */
   private SecurityActions()
   {
   }

   //-------------------------------------------------------------------------------------||
   // Utility Methods --------------------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   /**
    * Returns a new instance of the specified class, defined in the 
    * specified ClassLoader.  If the CL is null, the Thread Context CL
    * will be used.
    * 
    * @throws IllegalArgumentException If the class was not specified
    * @throws PrivilegedActionException 
    */
   static Object newInstance(final Class<?> clazz, final ClassLoader cl) throws IllegalArgumentException, Exception
   {

      // Whether to set the CL
      final boolean setDefiningCl = cl != null;

      // Make new instance
      return AccessController.doPrivileged(new PrivilegedExceptionAction<Object>()
      {

         public Object run() throws Exception
         {
            // Initialize
            Thread current = null;
            ClassLoader oldCl = null;

            // If necessary, set the TCCL
            if (setDefiningCl)
            {
               current = Thread.currentThread();
               oldCl = current.getContextClassLoader();
               current.setContextClassLoader(cl);
            }

            try
            {
               // Create the instance
               return clazz.newInstance();
            }
            finally
            {
               // Reset the TCCL if necessary
               if (setDefiningCl)
               {
                  current.setContextClassLoader(oldCl);
               }
            }
         }

      });
   }

   /**
    * Obtains the Thread Context ClassLoader
    * 
    * @return
    */
   static ClassLoader getTccl()
   {
      return AccessController.doPrivileged(new PrivilegedAction<ClassLoader>()
      {

         public ClassLoader run()
         {
            // Return the TCCL
            return Thread.currentThread().getContextClassLoader();
         }

      });
   }
}
