/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2009, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.bootstrap.api.server;

import java.util.EnumSet;

import org.jboss.bootstrap.api.config.InvalidConfigurationException;
import org.jboss.bootstrap.api.config.ServerConfig;
import org.jboss.bootstrap.api.lifecycle.LifecycleEventException;
import org.jboss.bootstrap.api.lifecycle.LifecycleEventHandler;
import org.jboss.bootstrap.api.lifecycle.LifecycleState;

/**
 * Server
 * 
 * Client view of generic Server implementations
 *
 * @author <a href="mailto:andrew.rubinger@jboss.org">ALR</a>
 * @version $Revision: $
 */
public interface Server<K extends Server<K, T>, T extends ServerConfig<T>>
{
   //-------------------------------------------------------------------------------------||
   // Contracts --------------------------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   /**
    * Prepares the server for startup.  If {@link Server#getServerInitializer()} is not 
    * null, will invoke {@link ServerInitializer#initialize(Server)}.
    * Freezes the configuration from further 
    * mutable actions by invoking {@link ServerConfig#freeze()}.  Finally
    * sets the current state to {@link LifecycleState#IDLE}
    * 
    * @throws IllegalStateException  If the server is not in state {@link LifecycleState#PRE_INIT}, 
    *       or if the configuration has not been defined
    * @throws InvalidConfigurationException
    * @throws LifecycleEventException If an error was encountered in the lifecycle handlers for this event
    */
   void initialize() throws IllegalStateException, InvalidConfigurationException, LifecycleEventException;

   /**
    * Start lifecycle of the Server, optionally invoking upon
    * {@link Server#initialize()} if the state is {@link LifecycleState#PRE_INIT}.
    * During execution the state will be {@link LifecycleState#STARTING}, and upon
    * successful completion will be {@link LifecycleState#STARTED}. 
    * 
    * @throws IllegalStateException If the Server is already started or starting
    * @throws Exception
    */
   void start() throws IllegalStateException, Exception;

   /**
    * Stop lifecycle of the Server.  During execution the state will be 
    * {@link LifecycleState#STOPPING}, and upon
    * successful completion will be {@link LifecycleState#IDLE}.  If a 
    * server initialzer is defined then {@link ServerInitializer#cleanup(Server)}
    * will be invoked during this call.
    * 
    * @throws IllegalStateException If the Server is not started
    * @throws Exception
    */
   void stop() throws IllegalStateException, Exception;

   /**
    * Alias for {@link Server#stop()}
    * 
    * @throws IllegalStateException If the Server is not started
    * @throws Exception
    */
   void shutdown() throws IllegalStateException, Exception;

   /**
    * Returns the current state of the Server lifecycle
    * 
    * @return The {@link LifecycleState}
    */
   LifecycleState getState();

   /**
    * Registers the specified handler to fire when 
    * the server's state enters that of the specified state.
    * 
    * @param state
    * @param handler
    * @throws IllegalArgumentException If either the state or the handler are unspecified
    */
   void registerEventHandler(LifecycleState state, LifecycleEventHandler handler) throws IllegalArgumentException;

   /**
    * Registers the specified handler to fire when 
    * the server's state enters one of the the specified states
    * 
    * @param handler
    * @param states
    * @throws IllegalArgumentException If either the states or the handler are unspecified
    */
   void registerEventHandler(LifecycleEventHandler handler, EnumSet<LifecycleState> states)
         throws IllegalArgumentException;

   /**
    * Registers the specified handler to fire when 
    * the server's state enters one of the the specified states
    * 
    * @param handler
    * @param states
    * @throws IllegalArgumentException If either the states or the handler are unspecified
    */
   void registerEventHandler(LifecycleEventHandler handler, LifecycleState... states) throws IllegalArgumentException;

   /**
    * Registers the specified handlers to fire when 
    * the server's state enters that of the specified state.
    * 
    * @param state
    * @param handlers
    * @throws IllegalArgumentException If either the state or the handlers are unspecified
    */
   void registerEventHandlers(LifecycleState state, LifecycleEventHandler... handlers) throws IllegalArgumentException;

   /**
    * Unregisters the specified event handler from firing 
    * when the server state changes to the specified state.
    * 
    * @param state
    * @param handler
    * @return Whether or not the handler was removed (ie. false if not registered)
    * @throws IllegalArgumentException If either the state or handler are unspecified
    */
   boolean unregisterEventHandler(LifecycleEventHandler handler, LifecycleState state) throws IllegalArgumentException;

   /**
    * Returns the underlying server configuration.  If
    * the server state is anything aside from 
    * {@link LifecycleState#PRE_INIT}, the configuration 
    * will be immutable / frozen.
    * 
    * @return
    */
   T getConfiguration();

   /**
    * Sets the configuration
    * 
    * @param config
    * @return This server
    */
   void setConfiguration(T config);
}
