/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2009, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.bootstrap.impl.as.config;

import java.io.IOException;
import java.net.URL;
import java.net.URLConnection;

import org.jboss.bootstrap.api.as.config.JBossASBasedServerConfig;
import org.jboss.bootstrap.api.config.InvalidConfigurationException;
import org.jboss.bootstrap.impl.base.config.AbstractBasicConfigurationValidator;
import org.jboss.bootstrap.spi.config.ConfigurationValidator;
import org.jboss.logging.Logger;

/**
 * JBossASBasedConfigurationValidator
 * 
 * Base validator for JBossAS Configurations
 *
 * @author <a href="mailto:andrew.rubinger@jboss.org">ALR</a>
 * @version $Revision: $
 */
public class JBossASBasedConfigurationValidator<T extends JBossASBasedServerConfig<T>>
      extends
         AbstractBasicConfigurationValidator<T> implements ConfigurationValidator<T>
{

   //-------------------------------------------------------------------------------||
   // Class Members ----------------------------------------------------------------||
   //-------------------------------------------------------------------------------||

   private static final Logger log = Logger.getLogger(JBossASBasedConfigurationValidator.class);

   //-------------------------------------------------------------------------------||
   // Overridden Implementations ---------------------------------------------------||
   //-------------------------------------------------------------------------------||

   /**
    * @see org.jboss.bootstrap.spi.config.AbstractBasicConfigurationValidator#validate(org.jboss.bootstrap.spi.config.ServerConfig)
    */
   @Override
   public void validate(final T config) throws InvalidConfigurationException
   {
      // Precondition check
      if (config == null)
      {
         throw new IllegalArgumentException("Configuration is required, but was not supplied");
      }

      // Log
      if (log.isTraceEnabled())
      {
         log.trace("Validating Configuration: " + config + " ...");
      }

      // Validate the super configuration
      super.validate(config);

      /*
       * Validate bootstrap properties
       */
      final String name = config.getBootstrapName();
      final URL home = config.getBootstrapHome();
      final URL direct = config.getBootstrapUrl();

      // If no direct bootstrap URL
      if (direct == null)
      {
         // Ensure we may construct this via home+name
         if (home == null || (name == null || name.length() == 0))
         {
            throw new InvalidConfigurationException(
                  "If the bootstrap URL is not specified, both the bootstrap home and name must be.");
         }
      }

      /*
       * The following properties must be specified
       */

      // jboss.home
      this.require(config.getJBossHome(), JBossASBasedServerConfig.PROP_KEY_JBOSSAS_HOME);
      // jboss.bind.address
      this.require(config.getBindAddress(), JBossASBasedServerConfig.PROP_KEY_JBOSSAS_BIND_ADDRESS);
      // jboss.server.name
      this.require(config.getServerName(), JBossASBasedServerConfig.PROP_KEY_JBOSSAS_SERVER_NAME);
      // jboss.lib.url
      this.require(config.getBootLibraryLocation(), JBossASBasedServerConfig.PROP_KEY_JBOSSAS_BOOT_LIBRARY_URL);
      // jboss.server.base.url
      this.require(config.getServerBaseLocation(), JBossASBasedServerConfig.PROP_KEY_JBOSSAS_SERVER_BASE_URL);
      // jboss.server.home.url
      this.require(config.getServerHomeLocation(), JBossASBasedServerConfig.PROP_KEY_JBOSSAS_SERVER_HOME_URL);
      // jboss.common.base.url
      this.require(config.getCommonBaseLocation(), JBossASBasedServerConfig.PROP_KEY_JBOSSAS_COMMON_BASE_URL);
      // jboss.common.lib.url
      this.require(config.getCommonLibLocation(), JBossASBasedServerConfig.PROP_KEY_JBOSSAS_COMMON_LIBRARY_URL);
      // jboss.server.log.dir
      this.require(config.getServerLogLocation(), JBossASBasedServerConfig.PROP_KEY_JBOSSAS_SERVER_LOG_DIR, false);
      // jboss.server.conf.url
      this.require(config.getServerConfLocation(), JBossASBasedServerConfig.PROP_KEY_JBOSSAS_SERVER_CONF_URL);
      // jboss.server.lib.url
      this.require(config.getServerLibLocation(), JBossASBasedServerConfig.PROP_KEY_JBOSSAS_SERVER_LIBRARY_URL);
      // jboss.server.data.dir
      this.require(config.getServerDataLocation(), JBossASBasedServerConfig.PROP_KEY_JBOSSAS_SERVER_DATA_DIR, false);
      // jboss.server.tmp.dir
      this.require(config.getServerTempLocation(), JBossASBasedServerConfig.PROP_KEY_JBOSSAS_SERVER_TEMP_DIR, false);
      // jboss.partition.name
      this.require(config.getPartitionName(), JBossASBasedServerConfig.PROP_KEY_JBOSSAS_PARTITION_NAME);
      // jboss.native.load
      this.require(config.isLoadNative(), JBossASBasedServerConfig.PROP_KEY_JBOSSAS_NATIVE_LOAD);
      // jboss.native.lib
      this.require(config.getNativeLibraryLocation(), JBossASBasedServerConfig.PROP_KEY_JBOSSAS_NATIVE_DIR, false);
      // jboss.platform.mbeanserver
      this.require(config.isUsePlatformMBeanServer(), JBossASBasedServerConfig.PROP_KEY_JBOSSAS_PLATFORM_MBEANSERVER);

      // Log
      log.debugf("Configuration is valid: %s", config);
   }

   //-------------------------------------------------------------------------------||
   // Internal Helper Methods ------------------------------------------------------||
   //-------------------------------------------------------------------------------||

   /**
    * Requires that the specified argument is both non-null and
    * non-empty. 
    * 
    * @throws InvalidConfigurationException If the specified argument is 
    *   either null or empty
    */
   protected void require(final String arg, final String propertyName) throws InvalidConfigurationException
   {
      // If not specified
      if (arg == null || arg.length() == 0)
      {
         // Tell 'em about it
         throw new InvalidConfigurationException(propertyName + " must be specified");
      }
   }

   /**
    * Requires that the specified argument is both non-null and
    * non-empty. 
    * 
    * @throws InvalidConfigurationException If the specified argument is 
    *   either null or empty
    */
   protected void require(final Boolean arg, final String propertyName) throws InvalidConfigurationException
   {
      // If not specified
      if (arg == null)
      {
         // Tell 'em about it
         throw new InvalidConfigurationException(propertyName + " must be specified");
      }
   }

   /**
    * Requires that the specified argument is both non-null and
    * pointing to an existent location
    * 
    * @throws InvalidConfigurationException If the specified argument is 
    *   either null or nonexistent
    */
   protected void require(final URL arg, final String propertyName) throws InvalidConfigurationException
   {
      this.require(arg, propertyName, true);
   }

   /**
    * Requires that the specified argument is both non-null and, optionally,
    * pointing to an existent location
    * 
    * @throws InvalidConfigurationException If the specified argument is 
    *   either null or nonexistent
    */
   protected void require(final URL arg, final String propertyName, final boolean exists)
         throws InvalidConfigurationException
   {
      // If not specified
      if (arg == null)
      {
         // Throw
         throw new InvalidConfigurationException(propertyName + " must be specified");
      }

      // Try to open a connection, to see if exists
      if (exists)
      {
         try
         {
            final URLConnection connection = arg.openConnection();
            connection.connect();
         }
         catch (IOException ioe)
         {
            throw new InvalidConfigurationException("Could not get a connection to the " + propertyName + ": "
                  + arg.toExternalForm(), ioe);
         }
      }
   }

}
